(function(){
  'use strict';

  // ES5 on purpose (some environments still choke on modern syntax).
  function byId(id){ return document.getElementById(id); }

  function fmtMoneyCAD(n){
    if (!isFinite(n)) return '—';
    try { return new Intl.NumberFormat('en-CA', { style:'currency', currency:'CAD' }).format(n); }
    catch(e){ return 'CAD ' + Number(n).toFixed(2); }
  }

  function fmtDateISO(d){
    var y = d.getFullYear();
    var m = String(d.getMonth()+1); if (m.length<2) m = '0'+m;
    var da = String(d.getDate()); if (da.length<2) da = '0'+da;
    return y + '-' + m + '-' + da;
  }

  function parseDateInput(val){
    if (!val) return null;
    var parts = val.split('-');
    if (parts.length !== 3) return null;
    var y = Number(parts[0]);
    var m = Number(parts[1]);
    var d = Number(parts[2]);
    if (!isFinite(y) || !isFinite(m) || !isFinite(d)) return null;
    return new Date(y, m-1, d);
  }

  function daysInMonth(y, m0){
    return new Date(y, m0+1, 0).getDate();
  }

  function addMonthsKeepingDay(date, months){
    var y = date.getFullYear();
    var m = date.getMonth() + months;
    var d = date.getDate();

    var ny = y + Math.floor(m / 12);
    var nm = m % 12;
    if (nm < 0) { nm += 12; ny -= 1; }

    var dim = daysInMonth(ny, nm);
    var nd = Math.min(d, dim);
    return new Date(ny, nm, nd);
  }

  function addYearsKeepingDay(date, years){
    return addMonthsKeepingDay(date, years * 12);
  }

  function nextDepositDate(currentDate, freq, semiState){
    var d = new Date(currentDate.getFullYear(), currentDate.getMonth(), currentDate.getDate());

    if (freq === 'weekly') { d.setDate(d.getDate() + 7); return d; }
    if (freq === 'biweekly') { d.setDate(d.getDate() + 14); return d; }
    if (freq === 'monthly') { return addMonthsKeepingDay(d, 1); }
    if (freq === 'quarterly') { return addMonthsKeepingDay(d, 3); }
    if (freq === 'yearly') { return addYearsKeepingDay(d, 1); }

    // Semi-monthly: alternates 15th and last day, then 15th next month, etc.
    if (freq === 'semimonthly') {
      var y = d.getFullYear();
      var m = d.getMonth();
      var day = d.getDate();
      var last = daysInMonth(y, m);
      var isOn15 = (day === 15);
      var isOnLast = (day === last);

      if (!semiState) semiState = { next: null };

      if (semiState.next === null) {
        if (day <= 15) {
          semiState.next = isOn15 ? 'last' : '15';
        } else {
          semiState.next = isOnLast ? '15next' : 'last';
        }
      }

      if (semiState.next === '15') {
        d = new Date(y, m, 15);
        semiState.next = 'last';
        return d;
      }
      if (semiState.next === 'last') {
        d = new Date(y, m, last);
        semiState.next = '15next';
        return d;
      }

      // 15next
      d = addMonthsKeepingDay(new Date(y, m, 15), 1);
      d = new Date(d.getFullYear(), d.getMonth(), 15);
      semiState.next = 'last';
      return d;
    }

    return d;
  }

  var ppyMap = { weekly:52, biweekly:26, semimonthly:24, monthly:12, quarterly:4, yearly:1 };

  function requiredDeposit(years, goal, rate, ppy){
    var n = Math.round(years * ppy);
    var r = (rate/100) / ppy;
    if (n <= 0) return { dep:0, n:n, r:r };
    if (Math.abs(r) < 1e-12) return { dep:goal/n, n:n, r:0 };
    return { dep: goal / ((Math.pow(1+r, n) - 1) / r), n:n, r:r };
  }

  // Builds rows with exact terms + deposit date.
  function buildScheduleWithDates(n, r, dep, startDate, freq){
    var rows = [];
    var bal = 0;
    var totalDeposited = 0;
    var totalInterest = 0;

    var d = new Date(startDate.getFullYear(), startDate.getMonth(), startDate.getDate());
    var semiState = (freq === 'semimonthly') ? { next: null } : null;

    for (var i=1; i<=n; i++){
      var totalUntilNow = bal; // at start of period
      var interestGained = bal * r; // this period
      totalInterest += interestGained;
      bal += interestGained;

      bal += dep;
      totalDeposited += dep;

      rows.push({
        sn: i,
        depositDate: fmtDateISO(d),
        thisDeposit: dep,
        totalUntilNow: totalUntilNow,
        totalDeposited: totalDeposited,
        interestGained: interestGained,
        totalUntilNowWithAddedInterest: bal
      });

      d = nextDepositDate(d, freq, semiState);
    }

    return {
      rows: rows,
      ending: bal,
      totalDeposited: totalDeposited,
      totalInterest: totalInterest,
      nextDate: (n > 0 ? fmtDateISO(d) : fmtDateISO(startDate))
    };
  }

  function renderTable(tableId, metaId, rows){
    var tb = document.querySelector('#' + tableId + ' tbody');
    if (!tb) return;
    tb.innerHTML = '';

    var cap = 5000;
    var show = rows.length > cap ? rows.slice(0, cap) : rows;

    for (var i=0; i<show.length; i++){
      var r = show[i];
      var tr = document.createElement('tr');
      tr.innerHTML = ''
        + '<td>' + r.sn + '</td>'
        + '<td>' + r.depositDate + '</td>'
        + '<td>' + fmtMoneyCAD(r.thisDeposit) + '</td>'
        + '<td>' + fmtMoneyCAD(r.totalUntilNow) + '</td>'
        + '<td>' + fmtMoneyCAD(r.totalDeposited) + '</td>'
        + '<td>' + fmtMoneyCAD(r.interestGained) + '</td>'
        + '<td>' + fmtMoneyCAD(r.totalUntilNowWithAddedInterest) + '</td>';
      tb.appendChild(tr);
    }

    var meta = byId(metaId);
    if (meta) {
      meta.textContent = rows.length.toLocaleString() + ' periods' + (rows.length > cap ? (' (showing first ' + cap.toLocaleString() + ')') : '');
    }
  }

  function exportScheduleCSV(rows, filename){
    if (!rows || !rows.length) return;
    var lines = ['sn,deposit date,this deposit,total until now,total deposited,interest gained,total until now with added interest'];
    for (var i=0; i<rows.length; i++){
      var r = rows[i];
      lines.push([
        r.sn,
        r.depositDate,
        r.thisDeposit,
        r.totalUntilNow,
        r.totalDeposited,
        r.interestGained,
        r.totalUntilNowWithAddedInterest
      ].join(','));
    }
    var csv = lines.join('\n');

    var a = document.createElement('a');
    a.href = 'data:text/csv;charset=utf-8,' + encodeURIComponent(csv);
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.parentNode.removeChild(a);
  }

  function setWarn(id, msg){
    var el = byId(id);
    if (!el) return;
    if (!msg) { el.style.display='none'; el.textContent=''; return; }
    el.style.display='block';
    el.textContent = String(msg);
  }

  // Storage for exports
  window.__goalSchedule = [];
  window.__fvSchedule = [];

  function goalCalculate(){
    setWarn('warn', '');

    var startVal = byId('startDate').value;
    var startDate = parseDateInput(startVal);
    var years = Number(byId('years').value);
    var goal = Number(byId('goal').value);
    var rate = Number(byId('rate').value);
    var freq = byId('freq').value;

    if (!startDate) { setWarn('warn', 'Start date is required.'); return; }

    var ppy = ppyMap[freq];
    if (!ppy || years < 0 || goal < 0 || !isFinite(rate)) {
      setWarn('warn', 'Invalid inputs.');
      return;
    }

    var rd = requiredDeposit(years, goal, rate, ppy);
    var sched = buildScheduleWithDates(rd.n, rd.r, rd.dep, startDate, freq);

    byId('deposit').textContent = fmtMoneyCAD(rd.dep);
    byId('totalDeposits').textContent = fmtMoneyCAD(sched.totalDeposited);
    byId('interest').textContent = fmtMoneyCAD(sched.totalInterest);
    byId('ending').textContent = fmtMoneyCAD(sched.ending);
    byId('goalPill').textContent = ppy + '/yr • ' + rd.n + ' periods';
    byId('nextDepositGoal').textContent = sched.nextDate || '—';

    renderTable('tbl', 'tableMeta', sched.rows);

    window.__goalSchedule = sched.rows;
    byId('exportBtn').disabled = (sched.rows.length === 0);
  }

  function fvCalculate(){
    setWarn('fvWarn', '');

    var startVal = byId('fvStartDate').value;
    var startDate = parseDateInput(startVal);
    var years = Number(byId('fvYears').value);
    var dep = Number(byId('fvDeposit').value);
    var rate = Number(byId('fvRate').value);
    var freq = byId('fvFreq').value;

    if (!startDate) { setWarn('fvWarn', 'Start date is required.'); return; }

    var ppy = ppyMap[freq];
    if (!ppy || years < 0 || dep < 0 || !isFinite(rate)) {
      setWarn('fvWarn', 'Invalid inputs.');
      return;
    }

    var n = Math.round(years * ppy);
    var r = (rate/100) / ppy;

    var sched = buildScheduleWithDates(n, r, dep, startDate, freq);

    byId('fvTotalDeposits').textContent = fmtMoneyCAD(sched.totalDeposited);
    byId('fvInterest').textContent = fmtMoneyCAD(sched.totalInterest);
    byId('fvEnding').textContent = fmtMoneyCAD(sched.ending);
    byId('fvPeriods').textContent = String(n);
    byId('fvPill').textContent = ppy + '/yr • ' + n + ' periods';
    byId('nextDepositFV').textContent = sched.nextDate || '—';

    renderTable('fvTbl', 'fvTableMeta', sched.rows);

    window.__fvSchedule = sched.rows;
    byId('fvExportBtn').disabled = (sched.rows.length === 0);
  }

  function showTab(which){
    if (which === 'goal') {
      byId('tabGoal').className = 'tab active';
      byId('tabFV').className = 'tab';
      byId('tabGoalBtn').className = 'tabbtn active';
      byId('tabFVBtn').className = 'tabbtn';
    } else {
      byId('tabFV').className = 'tab active';
      byId('tabGoal').className = 'tab';
      byId('tabFVBtn').className = 'tabbtn active';
      byId('tabGoalBtn').className = 'tabbtn';
    }
  }

  // ---- Tests ----
  function approxEqual(a, b, eps){
    if (eps == null) eps = 1e-6;
    return Math.abs(a - b) <= eps * Math.max(1, Math.abs(a), Math.abs(b));
  }

  function runTests(){
    var fails = [];

    (function(){
      var res = requiredDeposit(0, 1000, 5, 12);
      if (!(res.n === 0 && res.dep === 0)) fails.push('0 years => n=0, dep=0');
    })();

    (function(){
      var res = requiredDeposit(1, 1200, 0, 12);
      if (!(res.n === 12 && approxEqual(res.dep, 100))) fails.push('0% interest monthly 1yr => 100');
    })();

    (function(){
      var goal = 10000;
      var res = requiredDeposit(5, goal, 6, 12);
      var start = new Date(2025, 0, 1);
      var sched = buildScheduleWithDates(res.n, res.r, res.dep, start, 'monthly');
      if (!approxEqual(sched.ending, goal, 5e-4)) fails.push('Round-trip reaches goal');
    })();

    // Date stepping tests
    (function(){
      var d = new Date(2025, 0, 31);
      var n1 = nextDepositDate(d, 'monthly', null);
      var ok = (n1.getMonth() === 1) && (n1.getDate() >= 28 && n1.getDate() <= 29);
      if (!ok) fails.push('Monthly date stepping clamps end-of-month');
    })();

    (function(){
      var d = new Date(2025, 0, 31);
      var q = nextDepositDate(d, 'quarterly', null);
      var ok = (q.getFullYear() === 2025 && q.getMonth() === 3 && q.getDate() === 30);
      if (!ok) fails.push('Quarterly date stepping keeps calendar');
    })();

    (function(){
      var s = { next: null };
      var d0 = new Date(2025, 0, 10);
      var d1 = nextDepositDate(d0, 'semimonthly', s);
      var d2 = nextDepositDate(d1, 'semimonthly', s);
      var ok = (d1.getMonth()===0 && d1.getDate()===15) && (d2.getMonth()===0 && d2.getDate()===31);
      if (!ok) fails.push('Semi-monthly alternates 15 and last-day');
    })();

    if (fails.length) {
      try { console.warn('Tests failed:\n- ' + fails.join('\n- ')); } catch(e){}
    }
  }

  function setDefaultDates(){
    var now = new Date();
    var iso = fmtDateISO(now);
    if (byId('startDate') && !byId('startDate').value) byId('startDate').value = iso;
    if (byId('fvStartDate') && !byId('fvStartDate').value) byId('fvStartDate').value = iso;
  }

  function init(){
    // Tabs
    byId('tabGoalBtn').onclick = function(){ showTab('goal'); };
    byId('tabFVBtn').onclick = function(){ showTab('fv'); };

    // Actions
    byId('calcBtn').onclick = goalCalculate;
    byId('exportBtn').onclick = function(){ exportScheduleCSV(window.__goalSchedule, 'goal_schedule.csv'); };

    byId('fvCalcBtn').onclick = fvCalculate;
    byId('fvExportBtn').onclick = function(){ exportScheduleCSV(window.__fvSchedule, 'future_value_schedule.csv'); };

    setDefaultDates();
    runTests();

    // Initial calc for goal tab
    goalCalculate();
  }

  if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', init);
  else init();

})();
