<?php
// File: /reporting/downtime_visualization.php
// Description: D3.js drilldown and dimension bar charts for downtime data
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Downtime Visualization</title>
    <script src="https://d3js.org/d3.v7.min.js"></script>
    <style>
        body { font-family: sans-serif; margin: 20px; }
        h1, h2, h3, h4 { margin: 0 0 10px; }
        .chart { margin-bottom: 60px; }
        .bar { fill: steelblue; }
        .bar:hover { fill: darkorange; cursor: pointer; }
        .axis-label { font-size: 12px; }
        .back-button { margin-bottom: 10px; }
    </style>
</head>
<body>
<h1>Downtime Visualization</h1>

<!-- Drilldown Date Chart -->
<section class="chart" id="chart-date-section">
    <h2>Date Drilldown</h2>
    <button id="back-button" class="back-button" style="display:none;">Back</button>
    <div id="chart-date"></div>
</section>

<!-- Dimension Charts -->
<section id="dimensions">
    <h2>Dimension Charts</h2>
    <div id="chart-equipment_type" class="chart"><h3>By Equipment Type</h3></div>
    <div id="chart-equipment_unit" class="chart"><h3>By Equipment Unit</h3></div>
    <div id="chart-equipment_subtype" class="chart"><h3>By Component</h3></div>
    <div id="chart-downtime_type" class="chart"><h3>By Trigger</h3></div>
    <div id="chart-cause" class="chart"><h3>By Cause</h3></div>
    <div id="chart-feed" class="chart"><h3>By Feed</h3></div>
    <div id="chart-solution" class="chart"><h3>By Solution</h3></div>
    <div id="chart-category" class="chart"><h3>By Planned?</h3></div>
    <div id="chart-production_stopped" class="chart"><h3>By Producing?</h3></div>
</section>

<script>
    // The code now fetches data from your local API endpoint.
    fetch('http://192.168.2.18/api/downtime.php')
        .then(res => res.json())
        .then(rawData => {
            const data = rawData.map(d => {
                const [h, m, s] = d.total_downtime.split(':').map(Number);
                return { ...d, minutes: h * 60 + m };
            });

            // Nest data: year -> month -> day
            const nested = d3.group(data,
                d => d.date.slice(0,4),
                d => d.date.slice(5,7),
                d => d.date.slice(8,10)
            );

            let currentLevel = 0;
            const selectedPath = [];
            const backBtn = d3.select('#back-button');

            backBtn.on('click', () => {
                if (currentLevel > 0) {
                    selectedPath.pop(); currentLevel--; renderDateChart();
                }
                if (currentLevel === 0) backBtn.style('display', 'none');
            });

            function updateDateOnClick(event, d) {
                if (currentLevel < 2) {
                    selectedPath.push(d.key);
                    currentLevel++;
                    renderDateChart();
                }
            }

            function renderDateChart() {
                const container = d3.select('#chart-date');
                container.selectAll('*').remove();
                let entries = [];

                if (currentLevel === 0) {
                    entries = Array.from(nested.keys()).map(year => ({ key: year, value: d3.sum(Array.from(nested.get(year).values()), months => d3.sum(Array.from(months.values()), days => d3.sum(days, d => d.minutes))) }));
                    backBtn.style('display', 'none');
                } else if (currentLevel === 1) {
                    const yearMap = nested.get(selectedPath[0]);
                    entries = Array.from(yearMap.keys()).map(month => ({ key: month, value: d3.sum(Array.from(yearMap.get(month).values()), days => d3.sum(days, d => d.minutes)) }));
                    backBtn.style('display', 'inline');
                } else {
                    const dayMap = nested.get(selectedPath[0]).get(selectedPath[1]);
                    entries = Array.from(dayMap.keys()).map(day => ({ key: day, value: d3.sum(dayMap.get(day), d => d.minutes) }));
                    backBtn.style('display', 'inline');
                }

                // Sort entries to ensure consistent order (e.g., months in order)
                entries.sort((a, b) => d3.ascending(a.key, b.key));

                renderBarChart('#chart-date', entries, updateDateOnClick);
            }

            // Initial render
            renderDateChart();

            // Render dimension charts
            const dims = ['equipment_type','equipment_unit','equipment_subtype','downtime_type','cause','feed','solution','category','production_stopped'];
            dims.forEach(dim => {
                const dimData = Array.from(
                    d3.rollup(data, v => d3.sum(v, d => d.minutes), d => d[dim]),
                    ([key, value]) => ({ key, value })
                ).sort((a, b) => d3.descending(a.value, b.value)); // Sort by value
                renderBarChart('#chart-' + dim, dimData);
            });
        })
        .catch(err => console.error('Error loading data:', err));


    // Generic bar chart renderer
    function renderBarChart(selector, data, onClick) {
        const margin = { top: 20, right: 20, bottom: 70, left: 60 }; // Increased bottom margin
        const width = 800 - margin.left - margin.right;
        const height = 400 - margin.top - margin.bottom;

        const svg = d3.select(selector)
            .append('svg')
            .attr('width', '100%') // Make SVG responsive
            .attr('height', '100%')
            .attr('viewBox', `0 0 ${width + margin.left + margin.right} ${height + margin.top + margin.bottom}`)
            .append('g')
            .attr('transform', `translate(${margin.left},${margin.top})`);

        const x = d3.scaleBand()
            .domain(data.map(d => d.key))
            .range([0, width])
            .padding(0.2);
        const y = d3.scaleLinear()
            .domain([0, d3.max(data, d => d.value)])
            .nice()
            .range([height, 0]);

        svg.append('g')
            .attr('transform', `translate(0,${height})`)
            .call(d3.axisBottom(x))
            .selectAll('text')
            .attr('transform', 'rotate(-45)')
            .style('text-anchor', 'end');

        svg.append('g').call(d3.axisLeft(y));

        const bars = svg.selectAll('.bar')
            .data(data)
            .enter()
            .append('rect')
            .attr('class', 'bar')
            .attr('x', d => x(d.key))
            .attr('width', x.bandwidth())
            .attr('y', d => y(d.value))
            .attr('height', d => height - y(d.value));

        if (onClick) {
            bars.on('click', (event, d) => onClick(event, d));
        }
    }
</script>
</body>
</html>
