<?php
// login.php

// Include header (which now starts buffering)
include 'header.php';

// LDAP/AD configuration
$ldapHost = 'ldap://192.168.2.5:389';
$baseDN   = 'DC=FCCL,DC=local';

// Allowed security group DNs
$allowedGroups = [
    'CN=DowntimeReportingAdmin,OU=Security,OU=Groups,OU=Corp,DC=FCCL,DC=local',
    'CN=DowntimeReportingAccess,OU=Security,OU=Groups,OU=Corp,DC=FCCL,DC=local'
];

// Log file for tracking login errors
$logFile = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'login_errors.log';
function write_log($message) {
    global $logFile;
    @error_log(date('[Y-m-d H:i:s] ') . $message . PHP_EOL, 3, $logFile);
}

$error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    if ($username && $password) {
        $ldapConn = ldap_connect($ldapHost);
        if (!$ldapConn) {
            write_log("LDAP connection failed to $ldapHost");
            $error = 'Authentication service unavailable';
        } else {
            ldap_set_option($ldapConn, LDAP_OPT_PROTOCOL_VERSION, 3);
            ldap_set_option($ldapConn, LDAP_OPT_REFERRALS, 0);
            $userDN = "{$username}@fccl.local";

            if (@ldap_bind($ldapConn, $userDN, $password)) {
                $fullName = null;
                $groups   = [];
                $filter   = "(sAMAccountName={$username})";
                $attrs    = ['givenName','displayName','cn','memberOf'];

                $sr = ldap_search($ldapConn, $baseDN, $filter, $attrs);
                if ($sr) {
                    $entries = ldap_get_entries($ldapConn, $sr);
                    if ($entries['count'] > 0) {
                        $u = $entries[0];
                        // pick a name attribute
                        if (!empty($u['givenname'][0])) {
                            $fullName = $u['givenname'][0];
                        } elseif (!empty($u['displayname'][0])) {
                            $fullName = $u['displayname'][0];
                        } else {
                            $fullName = $u['cn'][0] ?? $username;
                        }
                        // collect group DNs
                        if (!empty($u['memberof'])) {
                            for ($i = 0; $i < $u['memberof']['count']; $i++) {
                                $groups[] = $u['memberof'][$i];
                            }
                        }
                    } else {
                        write_log("LDAP search returned no entries for {$username}");
                        $fullName = $username;
                    }
                } else {
                    write_log("LDAP search failed for {$username}: " . ldap_error($ldapConn));
                    $fullName = $username;
                }

                // check allowed groups
                $matched = array_intersect($groups, $allowedGroups);
                if (empty($matched)) {
                    write_log("Access denied for {$username}, no matching group membership.");
                    $error = 'Access denied';
                } else {
                    // extract CNs
                    $sgCns = [];
                    foreach ($matched as $dn) {
                        if (preg_match('/^CN=([^,]+)/', $dn, $m)) {
                            $sgCns[] = $m[1];
                        }
                    }

                    // set session
                    $_SESSION['username']       = $username;
                    $_SESSION['user_fullname']  = $fullName;
                    $_SESSION['user_sg_cns']    = $sgCns;
                    $_SESSION['loggedin']       = true;
                    $_SESSION['user'] = in_array('DowntimeReportingAdmin', $sgCns)
                        ? 'admin'
                        : 'user';

                    ldap_unbind($ldapConn);
                    header('Location: index.php');
                    exit;
                }
                ldap_unbind($ldapConn);
            } else {
                write_log("LDAP bind failed for {$userDN}: [" . ldap_errno($ldapConn) . '] ' . ldap_error($ldapConn));
                $error = 'Invalid credentials';
                ldap_unbind($ldapConn);
            }
        }
    } else {
        write_log('Login attempt without username or password.');
        $error = 'Please enter both username and password.';
    }
}
?>

<section class="login-wrapper">
    <div class="column is-full-mobile is-half-tablet is-one-third-desktop">
        <?php if ($error): ?>
            <div class="notification is-danger">
                <?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>

        <h1 class="title has-text-centered">Login</h1>

        <form action="login.php" method="post">
            <div class="field">
                <label class="label">Email</label>
                <div class="field has-addons">
                    <p class="control is-expanded">
                        <input
                                name="username"
                                class="input"
                                type="text"
                                placeholder="Your Fowler username"
                                required>
                    </p>
                    <p class="control">
                        <span class="button is-static">@fowler.ca</span>
                    </p>
                </div>
            </div>

            <div class="field">
                <label class="label">Password</label>
                <div class="control">
                    <input name="password" class="input" type="password" required>
                </div>
            </div>

            <div class="field">
                <div class="control">
                    <button type="submit" class="button is-primary is-fullwidth">
                        Log In
                    </button>
                </div>
            </div>
        </form>
    </div>
</section>
<?php include 'footer.php'; ?>
