<?php
global $pdo;
session_start();
require_once realpath(__DIR__ . '/..') . '/db.php';

//print_r($_SESSION);

$loggedIn = !empty($_SESSION['user']);
$userId = $loggedIn ? $_SESSION['user'] : '';

function createApiKey(PDO $pdo, $name, $userId)
{
    $key = bin2hex(random_bytes(20));
    $expiresAt = (new DateTime('+1 year'))->format('Y-m-d H:i:s');

    $sql = "INSERT INTO api_keys (`key`, name, expires_at, user_id) VALUES (:key, :name, :expires_at, :user_id)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':key' => $key,
        ':name' => $name,
        ':expires_at' => $expiresAt,
        ':user_id' => $userId
    ]);

    return $key;
}

function listApiKeys(PDO $pdo, $userId)
{
    $sql = "SELECT id, `key`, name, created_at, expires_at, revoked FROM api_keys WHERE user_id = :user_id ORDER BY created_at DESC";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':user_id' => $userId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

$error = '';
$success = '';
if ($loggedIn && $_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['name'])) {
    try {
        $newKey = createApiKey($pdo, $_POST['name'], $userId);
        $success = "New API key: <code>{$newKey}</code>";
    } catch (Exception $e) {
        $error = htmlspecialchars($e->getMessage());
    }
}

$allKeys = $loggedIn ? listApiKeys($pdo, $userId) : [];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>API Key Management</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bulma@0.9.4/css/bulma.min.css">
</head>
<body>
<section class="section">
    <div class="container">
        <h1 class="title">API Key Management</h1>

        <?php if ($error): ?>
            <div class="notification is-danger"><?= $error ?></div>
        <?php elseif ($success): ?>
            <div class="notification is-success"><?= $success ?></div>
        <?php endif; ?>

        <?php if ($loggedIn): ?>
            <div class="box">
                <form method="POST">
                    <div class="field has-addons">
                        <div class="control is-expanded">
                            <label for="apiKeyName">Key Name</label>
                            <input class="input" id="apiKeyName" name="name"
                                   placeholder="Key Name" required>
                        </div>
                        <div class="control">
                            <button class="button is-primary" type="submit">Generate</button>
                        </div>
                    </div>
                </form>
            </div>

            <div class="box">
                <h2 class="subtitle">Usage</h2>

                <h3 class="title is-6">Excel</h3>
                <pre><code>Name: X-API-KEY   Value: &lt;YOUR_API_KEY&gt;</code></pre>

                <h3 class="title is-6">PHP (cURL)</h3>
                <pre><code>$ch = curl_init('https://yourserver/api/downtime.php');
curl_setopt($ch, CURLOPT_HTTPHEADER, ['X-API-KEY: YOUR_KEY']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$data = json_decode(curl_exec($ch), true);
</code></pre>

                <h3 class="title is-6">Power BI</h3>
                <pre><code>let
  Source = Json.Document(Web.Contents("https://yourserver/api/downtime.php", [Headers=[#"X-API-KEY"="YOUR_KEY"]]))
in
  Source
</code></pre>
            </div>

            <h2 class="subtitle">Your Keys</h2>
            <table class="table is-striped is-fullwidth">
                <thead>
                <tr>
                    <th>ID</th>
                    <th>Name</th>
                    <th>Key</th>
                    <th>Created</th>
                    <th>Expires</th>
                    <th>Status</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($allKeys as $r):
                    $expired = (new DateTime() > new DateTime($r['expires_at']));
                    $status = $r['revoked'] ? 'Revoked' : ($expired ? 'Expired' : 'Active');
                    ?>
                    <tr>
                        <td><?= $r['id'] ?></td>
                        <td><?= htmlspecialchars($r['name']) ?></td>
                        <td><code><?= htmlspecialchars($r['key']) ?></code></td>
                        <td><?= $r['created_at'] ?></td>
                        <td><?= $r['expires_at'] ?></td>
                        <td><?= $status ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <div class="notification is-warning">Please log in.</div>
        <?php endif; ?>

    </div>
</section>
</body>
</html>
