<?php
// File: 206_downtime_table.php
// Displays logged downtime entries centered in a fluid container

// Start session and include header
if (session_status() === PHP_SESSION_NONE) session_start();
include 'header.php';

// Redirect to login if not authenticated
if (!isset($_SESSION['loggedin'])) {
    header('Location: login.php');
    exit;
}

require 'db.php'; // initializes $pdo
global $pdo;

// Fetch entries with lookups
$sql = <<<'SQL'
SELECT
  d.id,
  d.date,
  d.downtime_start,
  d.downtime_stop,
  d.total_downtime,
  et.name        AS equipment_type,
  eu.unit_number AS equipment_unit,
  es.name        AS component,
  dt.name        AS downtime_trigger,
  c.name         AS cause,
  f.name         AS feed,
  d.category,
  d.production_stopped,
  d.user,
  s.name         AS solution,
  d.remarks
FROM downtime_data d
JOIN equipment_type    et ON d.equipment_type_id    = et.id
JOIN equipment_unit    eu ON d.equipment_unit_id    = eu.id
JOIN equipment_subtype es ON d.equipment_subtype_id = es.id
JOIN downtime_type     dt ON d.downtime_type_id     = dt.id
JOIN cause             c  ON d.cause_id             = c.id
JOIN feeds             f  ON d.feed_id              = f.id
JOIN type_of_solution  s  ON d.solution_id          = s.id
ORDER BY d.date ASC, d.downtime_start ASC
SQL;
$rows = $pdo->query($sql)->fetchAll();
// if no rows, show a friendly message and stop
if (empty($rows)) {
    echo '<section class="section">
            <div class="container">
              <div class="notification is-info has-text-centered">
                No data yet
              </div>
            </div>
          </section>';
    include 'footer.php';
    exit;
}
// Group by date
$grouped = [];
foreach ($rows as $row) {
    $grouped[$row['date']][] = $row;
}
?>

<section class="section">
    <!-- Increase table width and allow remarks to wrap -->
    <!--    <style>-->
    <!--        -->
    <!--    </style>-->

    <div class="container is-fluid">

        <div class="modal" id="exportModal">
            <div class="modal-background"></div>
            <div class="modal-card">
                <header class="modal-card-head">
                    <p class="modal-card-title">Export Downtime Data</p>
                    <button class="delete" aria-label="close" id="closeModal"></button>
                </header>
                <section class="modal-card-body">
                    <form id="exportForm" method="GET" action="export_downtime_csv.php">
                        <div class="field">
                            <label class="label">Year *</label>
                            <div class="control">
                                <div class="select is-fullwidth">
                                    <select name="year" id="yearSelect" required>
                                        <option value="">-- Select Year --</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="field">
                            <label class="label">Month</label>
                            <div class="control">
                                <div class="select is-fullwidth">
                                    <select name="month" id="monthSelect">
                                        <option value="">-- Select Month --</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="field">
                            <label class="label">Day</label>
                            <div class="control">
                                <div class="select is-fullwidth">
                                    <select name="day" id="daySelect">
                                        <option value="">-- Select Day --</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </form>
                </section>
                <footer class="modal-card-foot">
                    <button class="button is-link" onclick="validateAndSubmit('csv')">Export CSV</button>
                    <button class="button is-info" onclick="validateAndSubmit('json')">Export JSON</button>
                    <button class="button" id="cancelModal">Cancel</button>
                </footer>

            </div>
        </div>

        <div class="level">
            <div class="level-left">
                <div class="level-item">
                    <h1 class="title">Logged Downtime Entries</h1>
                </div>
            </div>

            <div class="level-right">
                <div class="level-item">
                    <button class="button is-primary" id="exportButton">Export Data</button>
                </div>
            </div>
        </div>

        <div class="table-container">
            <table class="table is-striped is-hoverable is-fullwidth">
                <thead>
                <tr>
                    <th>Date</th>
                    <th>SN</th>
                    <th>Start</th>
                    <th>Stop</th>
                    <th>Total</th>
                    <th>Equipment</th>
                    <th>Unit</th>
                    <th>Component</th>
                    <th>Trigger</th>
                    <th>Cause</th>
                    <th>Feed</th>
                    <th>Solution</th>
                    <th>Planned?/Producing?</th>
                    <th>user</th>
                    <th>Remarks</th>
                    <?php if (!empty($_SESSION['user']) && $_SESSION['user'] === 'admin'): ?>
                    <th>Actions</th>
                    <?php endif; ?>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($grouped as $date => $entries): ?>
                    <?php $count = count($entries); ?>
                    <?php foreach ($entries as $i => $e): ?>
                        <?php
                        $classes = [];
                        if ($i === 0) $classes[] = 'date-group';
                        if ($i === $count - 1) $classes[] = 'group-end';
                        $plannedText = ($e['category'] === 'planned') ? 'yes' : 'no';
                        $prodText = ($e['production_stopped'] === 'yes') ? 'stopped' : 'running';
                        ?>
                        <tr class="<?= implode(' ', $classes) ?>">
                            <?php if ($i === 0): ?>
                                <td rowspan="<?= $count ?>"><?= htmlspecialchars($date) ?></td>
                            <?php endif; ?>
                            <td><?= $i + 1 ?></td>
                            <td><?= htmlspecialchars($e['downtime_start']) ?></td>
                            <td><?= htmlspecialchars($e['downtime_stop']) ?></td>
                            <td><?= htmlspecialchars($e['total_downtime']) ?></td>
                            <td><?= htmlspecialchars($e['equipment_type']) ?></td>
                            <td><?= htmlspecialchars($e['equipment_unit']) ?></td>
                            <td><?= htmlspecialchars($e['component']) ?></td>
                            <td><?= htmlspecialchars($e['downtime_trigger']) ?></td>
                            <td><?= htmlspecialchars($e['cause']) ?></td>
                            <td><?= htmlspecialchars($e['feed']) ?></td>
                            <td><?= htmlspecialchars($e['solution']) ?></td>
                            <td><?= $plannedText ?> / <?= $prodText ?></td>
                            <td><?= htmlspecialchars($e['user']) ?></td>
                            <td><?= htmlspecialchars($e['remarks']) ?></td>
                            <?php if (!empty($_SESSION['user']) && $_SESSION['user'] === 'admin'): ?>
                            <td><a class="button is-small is-link" href="206_downtime_form.php?id=<?= urlencode($e['id']) ?>">Edit</a></td>
                            <?php endif; ?>
                        </tr>
                    <?php endforeach; ?>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</section>
<script>
    let availableDates = {};

    // Fetch available dates from backend
    fetch('api/available_dates.php')
        .then(res => res.json())
        .then(data => {
            availableDates = data;
            populateYears();
        });

    // Populate Year dropdown
    function populateYears() {
        const yearSelect = document.getElementById('yearSelect');
        Object.keys(availableDates).forEach(year => {
            const option = document.createElement('option');
            option.value = year;
            option.textContent = year;
            yearSelect.appendChild(option);
        });
    }

    // Handle Year change → populate Months
    document.getElementById('yearSelect').addEventListener('change', function () {
        const year = this.value;
        const monthSelect = document.getElementById('monthSelect');
        const daySelect = document.getElementById('daySelect');

        monthSelect.innerHTML = '<option value="">-- Select Month --</option>';
        daySelect.innerHTML = '<option value="">-- Select Day --</option>';

        if (year && availableDates[year]) {
            Object.keys(availableDates[year]).forEach(month => {
                const option = document.createElement('option');
                option.value = month;
                const monthName = new Date(`${year}-${month}-01`).toLocaleString('default', {month: 'long'});
                option.textContent = monthName;
                monthSelect.appendChild(option);
            });
        }
    });

    // Handle Month change → populate Days
    document.getElementById('monthSelect').addEventListener('change', function () {
        const year = document.getElementById('yearSelect').value;
        const month = this.value;
        const daySelect = document.getElementById('daySelect');

        daySelect.innerHTML = '<option value="">-- Select Day --</option>';

        if (year && month && availableDates[year] && availableDates[year][month]) {
            availableDates[year][month].forEach(day => {
                const option = document.createElement('option');
                option.value = day;
                option.textContent = day;
                daySelect.appendChild(option);
            });
        }
    });

    // Validate and submit form
    function validateAndSubmit(format) {
        const form = document.getElementById('exportForm');
        const year = form.year.value;
        const month = form.month.value;
        const day = form.day.value;

        if (!year) {
            alert("Please select a year.");
            return;
        }

        if (day && !month) {
            alert("If you select a day, you must also select a month.");
            return;
        }

        // Set action based on export format
        if (format === 'csv') {
            form.action = 'export_downtime_csv.php';
        } else if (format === 'json') {
            form.action = 'export_downtime_json.php';
        }

        form.submit();
    }


    // Modal controls
    document.getElementById('exportButton').addEventListener('click', () => {
        document.getElementById('exportModal').classList.add('is-active');
    });

    document.getElementById('closeModal').addEventListener('click', () => {
        document.getElementById('exportModal').classList.remove('is-active');
    });

    document.getElementById('cancelModal').addEventListener('click', () => {
        document.getElementById('exportModal').classList.remove('is-active');
    });
</script>


<?php include 'footer.php'; ?>
