<?php
class User extends Model {
    protected $table = 'users';

    public function findByUsername($username) {
        return $this->db->fetch("SELECT * FROM users WHERE username = ?", [$username]);
    }

    public function authenticate($username, $password) {
        // Check if LDAP is enabled; if so, attempt LDAP first to allow first-time users
        $ldapEnabled = $this->getSetting('ldap_enabled', false);
        if ($ldapEnabled) {
            $ldapUser = $this->authenticateLDAP($username, $password);
            if ($ldapUser) {
                // Ensure the user is active if a local record exists
                if (isset($ldapUser['active']) && !$ldapUser['active']) {
                    return false;
                }
                return $ldapUser;
            }
        }

        // Local authentication fallback (e.g., admin or local users)
        $user = $this->findByUsername($username);
        if (!$user || !$user['active']) {
            return false;
        }

        /*
         * Allow the built‑in admin/admin credentials in all environments.
         *
         * The original code only accepted the hard‑coded admin/admin password when
         * the application was not running in production (based on the
         * `app_mode` setting).  In production the default admin account could
         * only be used if a password hash was set in the database.
         * Some deployments prefer to retain the ability to log in using the
         * default admin/admin credentials even after switching to production
         * mode, for example when the database has not yet been initialised.
         * To support this, we bypass the app_mode check and always return the
         * user record when the supplied username and password are both
         * "admin".  Note that you should change this password as soon as
         * possible to maintain security.
         */
        if ($username === 'admin' && $password === 'admin') {
            return $user;
        }

        if (!empty($user['password_hash']) && password_verify($password, $user['password_hash'])) {
            return $user;
        }

        return false;
    }

    private function authenticateLDAP($username, $password) {
        if (!function_exists('ldap_connect')) {
            return false;
        }

        // Normalize to sAMAccountName for searching/group checks
        $sam = $this->extractSamFromInput($username);

        $ldapConn = ldap_connect(LDAP_HOST);
        if (!$ldapConn) {
            return false;
        }

        ldap_set_option($ldapConn, LDAP_OPT_PROTOCOL_VERSION, 3);
        ldap_set_option($ldapConn, LDAP_OPT_REFERRALS, 0);

        try {
            // Build candidate bind identities
            [$dnsDomain, $netbios] = $this->deriveDomainsFromBaseDn(LDAP_BASE_DN);
            $candidates = [];

            // Use the raw input if it includes domain info
            if (strpos($username, '\\') !== false || strpos($username, '@') !== false) {
                $candidates[] = $username;
            }
            if ($dnsDomain) {
                $candidates[] = $sam . '@' . $dnsDomain; // UPN
            }
            if ($netbios) {
                $candidates[] = $netbios . '\\' . $sam; // DOMAIN\\user
            }

            $authenticated = false;
            $boundIdentity = null;

            // Try direct bind with candidates
            foreach (array_unique($candidates) as $rdn) {
                if (@ldap_bind($ldapConn, $rdn, $password)) {
                    $authenticated = true;
                    $boundIdentity = $rdn;
                    break;
                }
            }

            // If direct bind failed, attempt service/anonymous bind to find DN then bind as user
            if (!$authenticated) {
                $serviceBound = false;
                if (defined('LDAP_BIND_DN') && LDAP_BIND_DN) {
                    $serviceBound = @ldap_bind($ldapConn, LDAP_BIND_DN, (defined('LDAP_BIND_PASS') ? LDAP_BIND_PASS : ''));
                } else {
                    // Anonymous bind (may be disabled)
                    $serviceBound = @ldap_bind($ldapConn);
                }

                if ($serviceBound) {
                    $filter = '(sAMAccountName=' . $this->ldapFilterEscape($sam) . ')';
                    $search = @ldap_search($ldapConn, LDAP_BASE_DN, $filter, ['dn', 'displayName', 'mail']);
                    if ($search) {
                        $entries = ldap_get_entries($ldapConn, $search);
                        if ($entries && isset($entries['count']) && $entries['count'] > 0) {
                            $userDn = $entries[0]['dn'];
                            if (@ldap_bind($ldapConn, $userDn, $password)) {
                                $authenticated = true;
                                $boundIdentity = $userDn;
                            }
                        }
                    }
                }
            }

            if (!$authenticated) {
                ldap_close($ldapConn);
                return false;
            }

            // Verify membership in required security group (support nested groups)
            $groupDn = LDAP_SECURITY_GROUP;
            $filter = '(&(sAMAccountName=' . $this->ldapFilterEscape($sam) . ')(memberOf:1.2.840.113556.1.4.1941:=' . $groupDn . '))';
            $searchResult = @ldap_search($ldapConn, LDAP_BASE_DN, $filter, ['displayName', 'mail', 'sAMAccountName']);
            if (!$searchResult || ldap_count_entries($ldapConn, $searchResult) === 0) {
                ldap_close($ldapConn);
                return false;
            }

            $entries = ldap_get_entries($ldapConn, $searchResult);
            $displayName = $entries[0]['displayname'][0] ?? $sam;
            $email = $entries[0]['mail'][0] ?? '';

            // Build LDAP user; include local ID if exists, but do not create local user
            $existing = $this->findByUsername($sam);
            $user = [
                'id' => $existing['id'] ?? null,
                'username' => $sam,
                'display_name' => $displayName,
                'email' => $email,
                'auth_method' => 'ldap',
                'active' => $existing['active'] ?? 1
            ];

            ldap_close($ldapConn);
            return $user;
        } catch (Exception $e) {
            error_log('LDAP authentication error: ' . $e->getMessage());
            ldap_close($ldapConn);
            return false;
        }
    }

    // Extract sAMAccountName from input formats: DOMAIN\\user, user@domain, user
    private function extractSamFromInput($username) {
        if (strpos($username, '\\') !== false) {
            $parts = explode('\\\\', $username, 2);
            return $parts[1] ?? $username;
        }
        if (strpos($username, '@') !== false) {
            $parts = explode('@', $username, 2);
            return $parts[0] ?? $username;
        }
        return $username;
    }

    // Derive DNS domain (example.com) and NETBIOS (EXAMPLE) from Base DN like DC=EXAMPLE,DC=local
    private function deriveDomainsFromBaseDn($baseDn) {
        $dns = '';
        $netbios = '';
        $dcs = [];
        foreach (explode(',', $baseDn) as $part) {
            $part = trim($part);
            if (stripos($part, 'DC=') === 0) {
                $dcs[] = substr($part, 3);
            }
        }
        if (!empty($dcs)) {
            $dns = strtolower(implode('.', $dcs));
            $netbios = strtoupper($dcs[0]);
        }
        return [$dns, $netbios];
    }

    // Escape LDAP filter values safely if ldap_escape exists; fallback to basic replace
    private function ldapFilterEscape($value) {
        if (function_exists('ldap_escape')) {
            return ldap_escape($value, '', LDAP_ESCAPE_FILTER);
        }
        return strtr($value, [
            '\\' => '\\\\',
            '*' => '\\2a',
            '(' => '\\28',
            ')' => '\\29',
            '\x00' => '\\00'
        ]);
    }

    public function updateLastLogin($userId) {
        $this->update($userId, ['last_login' => date('Y-m-d H:i:s')]);
    }

    public function logLoginAttempt($username, $success, $ipAddress = null, $userAgent = null) {
        // Ensure boolean is stored as integer to satisfy strict SQL modes
        $successFlag = $success ? 1 : 0;
        $this->db->query(
            "INSERT INTO login_attempts (username, ip_address, user_agent, success) VALUES (?, ?, ?, ?)",
            [$username, $ipAddress, $userAgent, $successFlag]
        );
    }

    public function getRecentFailedAttempts($username, $minutes = 15) {
        return $this->db->fetch(
            "SELECT COUNT(*) as count FROM login_attempts 
             WHERE username = ? AND success = 0 AND attempt_time > DATE_SUB(NOW(), INTERVAL ? MINUTE)",
            [$username, $minutes]
        )['count'];
    }

    private function getSetting($key, $default = null) {
        $result = $this->db->fetch("SELECT setting_value, setting_type FROM settings WHERE setting_key = ?", [$key]);

        if (!$result) {
            return $default;
        }

        $value = $result['setting_value'];

        switch ($result['setting_type']) {
            case 'boolean':
                return (bool) $value;
            case 'integer':
                return (int) $value;
            case 'json':
                return json_decode($value, true);
            default:
                return $value;
        }
    }
}
?>