<?php
class Setting extends Model {
    protected $table = 'settings';
    
    public function getValue($key, $default = null) {
        $setting = $this->db->fetch("SELECT setting_value, setting_type FROM settings WHERE setting_key = ?", [$key]);
        
        if (!$setting) {
            return $default;
        }
        
        $value = $setting['setting_value'];
        
        switch ($setting['setting_type']) {
            case 'boolean':
                return (bool) $value;
            case 'integer':
                return (int) $value;
            case 'json':
                return json_decode($value, true);
            default:
                return $value;
        }
    }
    
    public function setValue($key, $value, $type = 'string') {
        // Convert value based on type
        switch ($type) {
            case 'boolean':
                $value = $value ? '1' : '0';
                break;
            case 'json':
                $value = json_encode($value);
                break;
            default:
                $value = (string) $value;
        }
        
        $existing = $this->db->fetch("SELECT id FROM settings WHERE setting_key = ?", [$key]);
        
        if ($existing) {
            return $this->db->query(
                "UPDATE settings SET setting_value = ?, setting_type = ?, updated_at = NOW() WHERE setting_key = ?",
                [$value, $type, $key]
            );
        } else {
            return $this->db->query(
                "INSERT INTO settings (setting_key, setting_value, setting_type) VALUES (?, ?, ?)",
                [$key, $value, $type]
            );
        }
    }
    
    public function getAll() {
        $settings = [];
        $results = $this->db->fetchAll("SELECT setting_key, setting_value, setting_type, description FROM settings ORDER BY setting_key");
        
        foreach ($results as $row) {
            $value = $row['setting_value'];
            
            switch ($row['setting_type']) {
                case 'boolean':
                    $value = (bool) $value;
                    break;
                case 'integer':
                    $value = (int) $value;
                    break;
                case 'json':
                    $value = json_decode($value, true);
                    break;
            }
            
            $settings[$row['setting_key']] = [
                'value' => $value,
                'type' => $row['setting_type'],
                'description' => $row['description']
            ];
        }
        
        return $settings;
    }
    
    public function updateMultiple($settings) {
        $this->db->beginTransaction();
        
        try {
            foreach ($settings as $key => $data) {
                $this->setValue($key, $data['value'], $data['type']);
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            throw $e;
        }
    }
}
?>