<?php
class Material extends Model {
    protected $table = 'materials';
    
    public function search($query, $page = 1, $perPage = ITEMS_PER_PAGE) {
        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT * FROM materials 
                WHERE (name LIKE ? OR code LIKE ? OR material_group LIKE ?) 
                AND active = 1 
                ORDER BY name 
                LIMIT {$perPage} OFFSET {$offset}";
        
        $searchTerm = "%{$query}%";
        $data = $this->db->fetchAll($sql, [$searchTerm, $searchTerm, $searchTerm]);
        
        $countSql = "SELECT COUNT(*) as count FROM materials 
                     WHERE (name LIKE ? OR code LIKE ? OR material_group LIKE ?) 
                     AND active = 1";
        
        $total = $this->db->fetch($countSql, [$searchTerm, $searchTerm, $searchTerm])['count'];
        
        return [
            'data' => $data,
            'total' => $total,
            'page' => $page,
            'perPage' => $perPage,
            'totalPages' => ceil($total / $perPage)
        ];
    }
    
    public function getGroups() {
        return $this->db->fetchAll(
            "SELECT DISTINCT material_group FROM materials WHERE active = 1 ORDER BY material_group"
        );
    }
    
    public function findByCode($code) {
        return $this->db->fetch("SELECT * FROM materials WHERE code = ?", [$code]);
    }
    
    public function getActiveForLocation($locationId = null) {
        if ($locationId) {
            $sql = "SELECT m.*, 
                           CASE WHEN lm.material_id IS NOT NULL THEN 1 ELSE 0 END as selected,
                           lm.quantity, lm.unit, lm.notes
                    FROM materials m
                    LEFT JOIN location_materials lm ON m.id = lm.material_id AND lm.location_id = ?
                    WHERE m.active = 1
                    ORDER BY m.name";
            return $this->db->fetchAll($sql, [$locationId]);
        } else {
            return $this->findAll(['active' => 1], 'name');
        }
    }
    
    public function getUsageStats($materialId) {
        return $this->db->fetch(
            "SELECT COUNT(lm.location_id) as location_count,
                    SUM(lm.quantity) as total_quantity
             FROM location_materials lm
             WHERE lm.material_id = ?",
            [$materialId]
        );
    }
    
    public function create($data) {
        // Add created_by only if the session user exists in local users table
        if (isset($_SESSION['username'])) {
            $exists = $this->db->fetch("SELECT username FROM users WHERE username = ?", [$_SESSION['username']]);
            if ($exists) {
                $data['created_by'] = $_SESSION['username'];
            }
        }
        
        return parent::create($data);
    }
    
    public function validateData($data, $id = null) {
        $errors = [];
        
        if (empty($data['code'])) {
            $errors['code'] = 'Material code is required';
        } else {
            // Check for duplicate code
            $existing = $this->findByCode($data['code']);
            if ($existing && (!$id || $existing['id'] != $id)) {
                $errors['code'] = 'Material code already exists';
            }
        }
        
        if (empty($data['name'])) {
            $errors['name'] = 'Material name is required';
        }
        
        if (empty($data['material_group'])) {
            $errors['material_group'] = 'Material group is required';
        }
        
        if (!isset($data['price']) || !is_numeric($data['price']) || $data['price'] < 0) {
            $errors['price'] = 'Valid price is required';
        }
        
        return $errors;
    }
}
?>