<?php
class Location extends Model {
    protected $table = 'locations';
    
    public function findNearby($latitude, $longitude, $radius = 100, $limit = 50) {
        $sql = "SELECT *, 
                (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * 
                cos(radians(longitude) - radians(?)) + sin(radians(?)) * 
                sin(radians(latitude)))) AS distance 
                FROM locations 
                WHERE active = 1
                HAVING distance < ? 
                ORDER BY distance 
                LIMIT ?";
        
        return $this->db->fetchAll($sql, [$latitude, $longitude, $latitude, $radius, $limit]);
    }
    
    public function findByAddress($address) {
        return $this->db->fetchAll(
            "SELECT * FROM locations 
             WHERE (address_line1 LIKE ? OR city LIKE ? OR province LIKE ?) 
             AND active = 1 
             ORDER BY name",
            ["%{$address}%", "%{$address}%", "%{$address}%"]
        );
    }
    
    public function getWithMaterials($locationId) {
        $location = $this->find($locationId);
        if (!$location) {
            return null;
        }
        
        $materials = $this->db->fetchAll(
            "SELECT m.*, lm.quantity, lm.unit, lm.notes
             FROM materials m
             JOIN location_materials lm ON m.id = lm.material_id
             WHERE lm.location_id = ? AND m.active = 1
             ORDER BY m.name",
            [$locationId]
        );
        
        $location['materials'] = $materials;
        return $location;
    }
    
    public function updateMaterials($locationId, $materials) {
        $this->db->beginTransaction();
        
        try {
            // Remove existing materials
            $this->db->query("DELETE FROM location_materials WHERE location_id = ?", [$locationId]);
            
            // Add new materials
            foreach ($materials as $materialId => $data) {
                if (!empty($data['selected'])) {
                    $this->db->query(
                        "INSERT INTO location_materials (location_id, material_id, quantity, unit, notes) 
                         VALUES (?, ?, ?, ?, ?)",
                        [
                            $locationId,
                            $materialId,
                            $data['quantity'] ?? 0,
                            $data['unit'] ?? 'units',
                            $data['notes'] ?? ''
                        ]
                    );
                }
            }
            
            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            throw $e;
        }
    }
    
    public function toggleStatus($id) {
        $location = $this->find($id);
        if (!$location) {
            return false;
        }
        
        $newStatus = $location['active'] ? 0 : 1;
        return $this->update($id, ['active' => $newStatus]);
    }
    
    public function create($data) {
        // Add created_by only if the session user exists in local users table
        if (isset($_SESSION['username'])) {
            $exists = $this->db->fetch("SELECT username FROM users WHERE username = ?", [$_SESSION['username']]);
            if ($exists) {
                $data['created_by'] = $_SESSION['username'];
            }
            // If not exists, leave created_by unset (NULL) to satisfy FK
        }
        
        return parent::create($data);
    }
    
    public function validateData($data) {
        $errors = [];
        
        if (empty($data['name'])) {
            $errors['name'] = 'Location name is required';
        }
        
        if (empty($data['address_line1'])) {
            $errors['address_line1'] = 'Address is required';
        }
        
        if (empty($data['city'])) {
            $errors['city'] = 'City is required';
        }
        
        if (empty($data['province'])) {
            $errors['province'] = 'Province is required';
        }
        
        if (empty($data['postal_code'])) {
            $errors['postal_code'] = 'Postal code is required';
        }
        
        if (!isset($data['latitude']) || !is_numeric($data['latitude'])) {
            $errors['latitude'] = 'Valid latitude is required';
        }
        
        if (!isset($data['longitude']) || !is_numeric($data['longitude'])) {
            $errors['longitude'] = 'Valid longitude is required';
        }
        
        return $errors;
    }
    
    public function search($query, $page = 1, $perPage = ITEMS_PER_PAGE) {
        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT * FROM locations 
                WHERE (name LIKE ? OR address_line1 LIKE ? OR city LIKE ? OR province LIKE ?) 
                AND active = 1 
                ORDER BY name 
                LIMIT {$perPage} OFFSET {$offset}";
        
        $searchTerm = "%{$query}%";
        $data = $this->db->fetchAll($sql, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        
        $countSql = "SELECT COUNT(*) as count FROM locations 
                     WHERE (name LIKE ? OR address_line1 LIKE ? OR city LIKE ? OR province LIKE ?) 
                     AND active = 1";
        
        $total = $this->db->fetch($countSql, [$searchTerm, $searchTerm, $searchTerm, $searchTerm])['count'];
        
        return [
            'data' => $data,
            'total' => $total,
            'page' => $page,
            'perPage' => $perPage,
            'totalPages' => ceil($total / $perPage)
        ];
    }
}
?>