<?php
session_start();
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'core/Controller.php';
require_once 'core/Model.php';
require_once 'helpers/UrlHelper.php';

// Auto-load classes
spl_autoload_register(function ($class) {
    $paths = [
        'controllers/',
        'models/',
        'core/',
        'helpers/'
    ];
    foreach ($paths as $path) {
        $file = $path . $class . '.php';
        if (file_exists($file)) {
            require_once $file;
            return;
        }
    }
});

// Route params
$page   = $_GET['page'] ?? '';
$action = $_GET['action'] ?? '';
$id     = $_GET['id'] ?? '';

// ========================
// GLOBAL AUTH GUARD
// ========================
// Only the login page is public. Everything else requires a session.
// (If you want to allow logout without a session, add 'logout' to $publicPages.)
$publicPages = ['login', 'releases'];
if (!isset($_SESSION['user_id']) && !in_array($page, $publicPages, true)) {
    header('Location: ' . BASE_URL . '?page=login');
    exit;
}

// Debug
if (isset($_GET['debug'])) {
    echo "<pre>";
    echo "Page: " . $page . "\n";
    echo "Action: " . $action . "\n";
    echo "ID: " . $id . "\n";
    echo "GET params: ";
    print_r($_GET);
    echo "POST params: ";
    print_r($_POST);
    echo "</pre>";
}

try {
    // Route handling
    switch ($page) {
        case '':
        case 'home':
            $controller = new HomeController();
            if ($action === 'search') {
                $controller->search();
            } else {
                $controller->index();
            }
            break;

        case 'how-to-use':
            $controller = new HomeController();
            $controller->howToUse();
            break;

        case 'releases':
            $controller = new HomeController();
            $controller->releases();
            break;

        case 'login':
        case 'admin':
            $controller = new AuthController();
            $controller->login();
            break;

        case 'logout':
            $controller = new AuthController();
            $controller->logout();
            break;

        case 'materials':
            $controller = new MaterialController();
            switch ($action) {
                case 'add':
                    $controller->add();
                    break;
                case 'edit':
                    if ($id) {
                        $controller->edit($id);
                    } else {
                        $controller->index();
                    }
                    break;
                case 'delete':
                    if ($id) {
                        $controller->delete($id);
                    } else {
                        $controller->index();
                    }
                    break;
                default:
                    $controller->index();
                    break;
            }
            break;

        case 'locations':
            $controller = new LocationController();
            switch ($action) {
                case 'add':
                    $controller->add();
                    break;
                case 'edit':
                    if ($id) {
                        $controller->edit($id);
                    } else {
                        $controller->index();
                    }
                    break;
                case 'toggle':
                    if ($id) {
                        $controller->toggle($id);
                    } else {
                        $controller->index();
                    }
                    break;
                case 'materials':
                    if ($id) {
                        $controller->materials($id);
                    } else {
                        http_response_code(400);
                        echo json_encode(['error' => 'Missing id']);
                    }
                    break;
                default:
                    $controller->index();
                    break;
            }
            break;

        case 'settings':
            $controller = new SettingsController();
            if ($action === 'update') {
                $controller->update();
            } else {
                $controller->index();
            }
            break;

        case 'api':
            $controller = new ApiController();
            switch ($action) {
                case 'distance':
                    $controller->distance();
                    break;
                case 'geocode':
                    $controller->geocode();
                    break;
                case 'materials':
                    $controller->materials();
                    break;
                default:
                    http_response_code(404);
                    echo json_encode(['error' => 'API endpoint not found']);
                    break;
            }
            break;

        default:
            // 404 Not Found
            http_response_code(404);
            include 'views/errors/404.php';
            break;
    }

} catch (Exception $e) {
    // Error handling
    error_log("Application Error: " . $e->getMessage());

    if (defined('APP_DEBUG') && APP_DEBUG) {
        echo "<div style='background: #f8d7da; color: #721c24; padding: 20px; margin: 20px; border: 1px solid #f5c6cb; border-radius: 5px;'>";
        echo "<h3>Application Error</h3>";
        echo "<p><strong>Message:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
        echo "<p><strong>File:</strong> " . htmlspecialchars($e->getFile()) . "</p>";
        echo "<p><strong>Line:</strong> " . $e->getLine() . "</p>";
        echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
        echo "</div>";
    } else {
        echo "<div style='text-align: center; padding: 50px;'>";
        echo "<h2>Something went wrong</h2>";
        echo "<p>Please try again later.</p>";
        echo "<a href='" . BASE_URL . "'>Go Home</a>";
        echo "</div>";
    }
}
