<?php
class Router {
    private $routes = [];
    
    public function add($route, $controller) {
        $this->routes[$route] = $controller;
    }
    
    public function dispatch() {
        $uri = $this->getUri();
        
        // Debug information (remove in production)
        if (isset($_GET['debug'])) {
            echo "<pre>";
            echo "Request URI: " . $_SERVER['REQUEST_URI'] . "\n";
            echo "Parsed URI: " . $uri . "\n";
            echo "Base URL: " . BASE_URL . "\n";
            echo "Available routes:\n";
            print_r($this->routes);
            echo "</pre>";
        }
        
        // Check for exact match first
        if (isset($this->routes[$uri])) {
            $this->callController($this->routes[$uri]);
            return;
        }
        
        // Check for parameterized routes
        foreach ($this->routes as $route => $controller) {
            if ($this->matchRoute($route, $uri)) {
                $this->callController($controller, $this->extractParams($route, $uri));
                return;
            }
        }
        
        // 404 Not Found
        $this->show404();
    }
    
    private function getUri() {
        $uri = $_SERVER['REQUEST_URI'];
        
        // Remove query string
        if (($pos = strpos($uri, '?')) !== false) {
            $uri = substr($uri, 0, $pos);
        }
        
        $uri = trim($uri, '/');
        
        // Remove base path if exists
        $basePath = trim(parse_url(BASE_URL, PHP_URL_PATH), '/');
        if ($basePath && strpos($uri, $basePath) === 0) {
            $uri = substr($uri, strlen($basePath));
            $uri = trim($uri, '/');
        }
        
        return $uri;
    }
    
    private function matchRoute($route, $uri) {
        $routePattern = preg_replace('/\{[^}]+\}/', '([^/]+)', $route);
        $routePattern = '#^' . $routePattern . '$#';
        return preg_match($routePattern, $uri);
    }
    
    private function extractParams($route, $uri) {
        $routePattern = preg_replace('/\{([^}]+)\}/', '(?P<$1>[^/]+)', $route);
        $routePattern = '#^' . $routePattern . '$#';
        
        if (preg_match($routePattern, $uri, $matches)) {
            return array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
        }
        
        return [];
    }
    
    private function callController($controllerAction, $params = []) {
        list($controllerName, $method) = explode('@', $controllerAction);
        
        if (!class_exists($controllerName)) {
            $this->show404();
            return;
        }
        
        $controller = new $controllerName();
        
        if (!method_exists($controller, $method)) {
            $this->show404();
            return;
        }
        
        call_user_func_array([$controller, $method], $params);
    }
    
    private function show404() {
        http_response_code(404);
        include 'views/errors/404.php';
    }
}
?>