<?php
class Controller {
    protected $db;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->checkMaintenanceMode();
    }

    protected function view($view, $data = []) {
        extract($data);

        // Check if user is logged in
        $isLoggedIn = isset($_SESSION['user_id']);
        $user = $isLoggedIn ? $_SESSION : null;

        // Get settings
        $settings = $this->getSettings();

        include 'views/layout/header.php';
        include "views/{$view}.php";
        include 'views/layout/footer.php';
    }

    protected function json($data, $status = 200) {
        http_response_code($status);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }

    protected function redirect($url) {
        // Handle different URL formats
        if (strpos($url, '?') !== false) {
            // URL already has parameters
            header("Location: " . BASE_URL . $url);
        } elseif (strpos($url, '=') !== false) {
            // URL is a query string
            header("Location: " . BASE_URL . "?" . $url);
        } else {
            // Convert simple page names to query format
            if (empty($url)) {
                header("Location: " . BASE_URL);
            } else {
                header("Location: " . BASE_URL . "?page=" . $url);
            }
        }
        exit;
    }

    protected function requireAuth() {
        if (!isset($_SESSION['user_id'])) {
            $this->redirect('login');
        }

        // Check session timeout
        if (isset($_SESSION['last_activity']) &&
            (time() - $_SESSION['last_activity']) > SESSION_TIMEOUT) {
            session_destroy();
            $this->redirect('login?timeout=1');
        }

        $_SESSION['last_activity'] = time();
    }

    /**
     * Require that the current user is an administrator.
     *
     * This method first ensures that the user is authenticated, then
     * checks if the user has admin privileges. Users authenticated via LDAP
     * who are members of the admin security group are considered admins.
     * Local users (except the limited "mapd" account) are also considered admins.
     * The "mapd" account is specifically blocked from admin functions.
     */
    protected function requireAdmin() {
        $this->requireAuth();
        
        // Block the limited "mapd" account from admin functions
        if (isset($_SESSION['username']) && strtolower($_SESSION['username']) === 'mapd') {
            header('Location: ' . BASE_URL);
            exit;
        }
        
        // Check if user has admin privileges
        if (!$this->isAdmin()) {
            header('Location: ' . BASE_URL . '?error=' . urlencode('Access denied. Administrator privileges required.'));
            exit;
        }
    }
    
    /**
     * Check if the current user has administrator privileges.
     * 
     * @return bool True if user is an admin, false otherwise
     */
    protected function isAdmin() {
        if (!isset($_SESSION['user_id']) || !isset($_SESSION['username'])) {
            return false;
        }
        
        // Block the limited "mapd" account explicitly
        if (strtolower($_SESSION['username']) === 'mapd') {
            return false;
        }
        
        // Check authentication method
        $authMethod = $_SESSION['auth_method'] ?? 'local';
        
        if ($authMethod === 'ldap') {
            // For LDAP users, they must be members of the admin security group
            // If they successfully authenticated via LDAP, they're already verified
            // as members of the required group during the authentication process
            return true;
        } else {
            // For local users, only allow the built-in "admin" account
            // All other local accounts (including "mapd") are not admins
            return strtolower($_SESSION['username']) === 'admin';
        }
    }

    protected function getSettings() {
        static $settings = null;

        if ($settings === null) {
            $stmt = $this->db->query("SELECT setting_key, setting_value, setting_type FROM settings");
            $settings = [];

            while ($row = $stmt->fetch()) {
                $value = $row['setting_value'];

                switch ($row['setting_type']) {
                    case 'boolean':
                        $value = (bool) $value;
                        break;
                    case 'integer':
                        $value = (int) $value;
                        break;
                    case 'json':
                        $value = json_decode($value, true);
                        break;
                }

                $settings[$row['setting_key']] = $value;
            }
        }

        return $settings;
    }

    protected function getSetting($key, $default = null) {
        $settings = $this->getSettings();
        return isset($settings[$key]) ? $settings[$key] : $default;
    }

    private function checkMaintenanceMode() {
        if ($this->getSetting('maintenance_mode', false) &&
            !isset($_SESSION['user_id'])) {
            include 'views/maintenance.php';
            exit;
        }
    }

    protected function logActivity($action, $details = null) {
        if (isset($_SESSION['username'])) {
            $this->db->query(
                "INSERT INTO user_activity (username, action, details, ip_address, user_agent) 
                 VALUES (?, ?, ?, ?, ?)",
                [
                    $_SESSION['username'],
                    $action,
                    $details,
                    $_SERVER['REMOTE_ADDR'] ?? null,
                    $_SERVER['HTTP_USER_AGENT'] ?? null
                ]
            );
        }
    }
}
?>