<?php
class SettingsController extends Controller {
    private $settingModel;

    public function __construct() {
        parent::__construct();
        // Only admins should access settings
        $this->requireAdmin();
        $this->settingModel = class_exists('Setting') ? new Setting() : null;
    }

    public function index() {
        $rawRows = $this->getAllSettings();
        $rows    = $this->normalizeRows($rawRows);   // ensure assoc rows: setting_key/value/type

        // Build simple key => value map for easy use in the view
        $settingsMap = [];
        foreach ($rows as $r) {
            if ($r['setting_key'] !== null) {
                $settingsMap[$r['setting_key']] = $r['setting_value'];
            }
        }

        $this->view('settings/index', [
            'dbSettings' => $rows,        // list of assoc rows for the table
            'settings'   => $settingsMap, // key => value map for quick lookups (logo, favicon, etc.)
            'success'    => $_GET['success'] ?? '',
            'error'      => $_GET['error'] ?? ''
        ]);
    }

    public function update() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('page=settings');
        }

        $form = $_POST['form'] ?? 'general';

        try {
            if ($form === 'branding') {
                $this->handleBrandingUpdate();
                $this->redirect('page=settings&success=' . urlencode('Branding updated.'));
            } else {
                // Generic key/value saves from the table
                $skip = ['form'];
                foreach ($_POST as $k => $v) {
                    if (in_array($k, $skip, true)) continue;
                    $this->upsertSetting($k, (string)$v, 'string');
                }
                $this->redirect('page=settings&success=' . urlencode('Settings saved.'));
            }
        } catch (Exception $e) {
            $this->redirect('page=settings&error=' . urlencode($e->getMessage()));
        }
    }

    /* ---------------- Branding helpers ---------------- */

    private function handleBrandingUpdate() {
        $root = dirname(__DIR__); // one up from controllers/
        $uploadDir = $root . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'branding';
        if (!is_dir($uploadDir)) {
            @mkdir($uploadDir, 0775, true);
        }

        $sanitizeUrl = function ($url) {
            $url = trim((string)$url);
            if ($url === '') return '';
            if (!preg_match('~^(https?://|/)~i', $url)) {
                throw new Exception('Logo/Favicon URL must be absolute (http/https) or site-relative (/...)');
            }
            return $url;
        };

        $saveFile = function ($field, $allowedExt, $basename) use ($uploadDir) {
            if (!isset($_FILES[$field]) || empty($_FILES[$field]['name'])) return null;
            if ($_FILES[$field]['error'] !== UPLOAD_ERR_OK) {
                throw new Exception("Upload error for $field (code " . (int)$_FILES[$field]['error'] . ")");
            }
            if (($_FILES[$field]['size'] ?? 0) > 2 * 1024 * 1024) {
                throw new Exception("File too large for $field (max 2 MB)");
            }

            $tmp  = $_FILES[$field]['tmp_name'];
            $map  = [
                'image/png'                => 'png',
                'image/x-icon'             => 'ico',
                'image/vnd.microsoft.icon' => 'ico',
                'image/jpeg'               => 'jpg',
                'image/svg+xml'            => 'svg',
            ];
            $finfo = function_exists('finfo_open') ? finfo_open(FILEINFO_MIME_TYPE) : null;
            $mime  = $finfo ? finfo_file($finfo, $tmp) : null;
            if ($finfo) finfo_close($finfo);
            $ext   = $map[$mime] ?? strtolower(pathinfo($_FILES[$field]['name'], PATHINFO_EXTENSION));

            if (!in_array($ext, $allowedExt, true)) {
                throw new Exception("Invalid file type for $field. Allowed: " . implode(', ', $allowedExt));
            }

            $target = $uploadDir . DIRECTORY_SEPARATOR . $basename . '.' . $ext;
            if (!move_uploaded_file($tmp, $target)) {
                throw new Exception("Failed to save $field");
            }

            return 'assets/branding/' . $basename . '.' . $ext; // site-relative path
        };

        // Inputs
        $logoUrl    = $sanitizeUrl($_POST['branding_logo_url'] ?? '');
        $faviconUrl = $sanitizeUrl($_POST['branding_favicon_url'] ?? '');

        // Files (file wins over URL if provided)
        $logoRel    = $saveFile('branding_logo_file',   ['png', 'jpg', 'jpeg', 'svg'], 'logo');
        $faviconRel = $saveFile('branding_favicon_file', ['ico', 'png'],               'favicon');

        if ($logoRel) {
            $this->upsertSetting('branding_logo', $logoRel, 'string');
        } elseif ($logoUrl !== '') {
            $this->upsertSetting('branding_logo', $logoUrl, 'string');
        }

        if ($faviconRel) {
            $this->upsertSetting('branding_favicon', $faviconRel, 'string');
        } elseif ($faviconUrl !== '') {
            $this->upsertSetting('branding_favicon', $faviconUrl, 'string');
        }
    }

    /* ---------------- DB helpers ---------------- */

    private function getAllSettings() {
        if ($this->settingModel && method_exists($this->settingModel, 'getAll')) {
            return $this->settingModel->getAll();
        }
        // May return numeric indexes depending on your DB layer; we normalize later.
        return $this->db->fetchAll("SELECT setting_key, setting_value, setting_type FROM settings ORDER BY setting_key");
    }

    private function normalizeRows(array $rows): array {
        $out = [];
        foreach ($rows as $r) {
            if (is_object($r)) $r = (array)$r;
            if (!is_array($r)) continue;

            // Be forgiving about key casing and numeric-indexed rows
            $lower = array_change_key_case($r, CASE_LOWER);

            $rk = $lower['setting_key']   ?? ($r['setting_key']   ?? ($r[0] ?? null));
            $rv = $lower['setting_value'] ?? ($r['setting_value'] ?? ($r[1] ?? null));
            $rt = $lower['setting_type']  ?? ($r['setting_type']  ?? ($r[2] ?? 'string'));

            $out[] = [
                'setting_key'   => $rk,
                'setting_value' => $rv,
                'setting_type'  => $rt,
            ];
        }
        return $out;
    }

    private function upsertSetting($key, $value, $type = 'string') {
        $sql = "INSERT INTO settings (setting_key, setting_value, setting_type)
                VALUES (?, ?, ?)
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), setting_type = VALUES(setting_type)";
        $this->db->query($sql, [$key, $value, $type]);
    }
}
