<?php
class MaterialController extends Controller {
    private $materialModel;

    public function __construct() {
        parent::__construct();
        // Restrict access: only full administrators can reach this controller
        $this->requireAdmin();
        $this->materialModel = new Material();
    }

    public function index() {
        $page = (int) ($_GET['p'] ?? 1); // Use 'p' for pagination to avoid conflict with 'page' route parameter
        $search = $_GET['search'] ?? '';

        // Handle inline form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $action = $_POST['action'] ?? '';

            if ($action === 'add') {
                $this->handleAdd();
                return;
            } elseif ($action === 'edit') {
                $id = $_POST['id'] ?? '';
                if ($id) {
                    $this->handleEdit($id);
                    return;
                }
            }
        }

        if ($search) {
            $result = $this->materialModel->search($search, $page);
        } else {
            $result = $this->materialModel->paginate($page, ITEMS_PER_PAGE, ['active' => 1], 'name');
        }

        $groups = $this->materialModel->getGroups();

        $this->view('materials/index', [
            'materials' => $result['data'],
            'pagination' => $result,
            'search' => $search,
            'groups' => $groups
        ]);
    }

    private function handleAdd() {
        $data = [
            'code' => trim($_POST['code'] ?? ''),
            'name' => trim($_POST['name'] ?? ''),
            'material_group' => trim($_POST['material_group'] ?? ''),
            'price' => (float) ($_POST['price'] ?? 0),
            'description' => trim($_POST['description'] ?? '')
        ];

        $errors = $this->materialModel->validateData($data);

        if (empty($errors)) {
            try {
                $id = $this->materialModel->create($data);
                $this->logActivity('material_created', "Created material: {$data['name']} ({$data['code']})");
                $this->redirect('page=materials&success=' . urlencode('Material added successfully'));
            } catch (Exception $e) {
                $this->redirect('page=materials&error=' . urlencode('Failed to create material: ' . $e->getMessage()));
            }
        } else {
            $errorMsg = implode(', ', $errors);
            $this->redirect('page=materials&error=' . urlencode($errorMsg));
        }
    }

    private function handleEdit($id) {
        $material = $this->materialModel->find($id);

        if (!$material) {
            $this->redirect('page=materials&error=' . urlencode('Material not found'));
            return;
        }

        $data = [
            'code' => trim($_POST['code'] ?? ''),
            'name' => trim($_POST['name'] ?? ''),
            'material_group' => trim($_POST['material_group'] ?? ''),
            'price' => (float) ($_POST['price'] ?? 0),
            'description' => trim($_POST['description'] ?? '')
        ];

        $errors = $this->materialModel->validateData($data, $id);

        if (empty($errors)) {
            try {
                $this->materialModel->update($id, $data);
                $this->logActivity('material_updated', "Updated material: {$data['name']} ({$data['code']})");
                $this->redirect('page=materials&success=' . urlencode('Material updated successfully'));
            } catch (Exception $e) {
                $this->redirect('page=materials&error=' . urlencode('Failed to update material: ' . $e->getMessage()));
            }
        } else {
            $errorMsg = implode(', ', $errors);
            $this->redirect('page=materials&error=' . urlencode($errorMsg));
        }
    }

    public function add() {
        $errors = [];
        $data = [];

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = [
                'code' => trim($_POST['code'] ?? ''),
                'name' => trim($_POST['name'] ?? ''),
                'material_group' => trim($_POST['material_group'] ?? ''),
                'price' => (float) ($_POST['price'] ?? 0),
                'description' => trim($_POST['description'] ?? '')
            ];

            $errors = $this->materialModel->validateData($data);

            if (empty($errors)) {
                try {
                    $id = $this->materialModel->create($data);
                    $this->logActivity('material_created', "Created material: {$data['name']} ({$data['code']})");
                    $this->redirect('page=materials&success=' . urlencode('Material added successfully'));
                } catch (Exception $e) {
                    $errors['general'] = 'Failed to create material: ' . $e->getMessage();
                }
            }
        }

        $groups = $this->materialModel->getGroups();

        $this->view('materials/form', [
            'material' => $data,
            'errors' => $errors,
            'groups' => $groups,
            'isEdit' => false
        ]);
    }

    public function edit($id) {
        $material = $this->materialModel->find($id);

        if (!$material) {
            $this->redirect('page=materials&error=' . urlencode('Material not found'));
        }

        $errors = [];

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = [
                'code' => trim($_POST['code'] ?? ''),
                'name' => trim($_POST['name'] ?? ''),
                'material_group' => trim($_POST['material_group'] ?? ''),
                'price' => (float) ($_POST['price'] ?? 0),
                'description' => trim($_POST['description'] ?? '')
            ];

            $errors = $this->materialModel->validateData($data, $id);

            if (empty($errors)) {
                try {
                    $this->materialModel->update($id, $data);
                    $this->logActivity('material_updated', "Updated material: {$data['name']} ({$data['code']})");
                    $this->redirect('page=materials&success=' . urlencode('Material updated successfully'));
                } catch (Exception $e) {
                    $errors['general'] = 'Failed to update material: ' . $e->getMessage();
                }
            } else {
                $material = array_merge($material, $data);
            }
        }

        $groups = $this->materialModel->getGroups();

        $this->view('materials/form', [
            'material' => $material,
            'errors' => $errors,
            'groups' => $groups,
            'isEdit' => true
        ]);
    }

    public function delete($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('page=materials');
        }

        $material = $this->materialModel->find($id);

        if (!$material) {
            $this->redirect('page=materials&error=' . urlencode('Material not found'));
        }

        try {
            // Check if material is used in any locations
            $usage = $this->materialModel->getUsageStats($id);

            if ($usage['location_count'] > 0) {
                // Soft delete - mark as inactive
                $this->materialModel->update($id, ['active' => 0]);
                $message = 'Material deactivated (was in use at ' . $usage['location_count'] . ' location(s))';
            } else {
                // Hard delete
                $this->materialModel->delete($id);
                $message = 'Material deleted successfully';
            }

            $this->logActivity('material_deleted', "Deleted material: {$material['name']} ({$material['code']})");
            $this->redirect('page=materials&success=' . urlencode($message));
        } catch (Exception $e) {
            $this->redirect('page=materials&error=' . urlencode('Failed to delete material: ' . $e->getMessage()));
        }
    }
}
?>