<?php
class LocationController extends Controller {
    private $locationModel;
    private $materialModel;

    public function __construct() {
        parent::__construct();
        // Restrict access: only full administrators can reach this controller
        $this->requireAdmin();
        $this->locationModel = new Location();
        $this->materialModel = new Material();
    }

    public function index() {
        $page = (int) ($_GET['p'] ?? 1); // Use 'p' for pagination to avoid conflict with 'page' route parameter
        $search = $_GET['search'] ?? '';

        // Handle inline form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $action = $_POST['action'] ?? '';

            if ($action === 'add') {
                $this->handleAdd();
                return;
            } elseif ($action === 'edit') {
                $id = $_POST['id'] ?? '';
                if ($id) {
                    $this->handleEdit($id);
                    return;
                }
            }
        }

        if ($search) {
            $result = $this->locationModel->search($search, $page);
        } else {
            $result = $this->locationModel->paginate($page, ITEMS_PER_PAGE, [], 'name');
        }

        $materials = $this->materialModel->getActiveForLocation();

        $this->view('locations/index', [
            'locations' => $result['data'],
            'pagination' => $result,
            'search' => $search,
            'materials' => $materials,
            'googleMapsApiKey' => $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY)
        ]);
    }

    private function handleAdd() {
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'address_line1' => trim($_POST['address_line1'] ?? ''),
            'address_line2' => trim($_POST['address_line2'] ?? ''),
            'city' => trim($_POST['city'] ?? ''),
            'province' => trim($_POST['province'] ?? ''),
            'postal_code' => trim($_POST['postal_code'] ?? ''),
            'country' => trim($_POST['country'] ?? 'Canada'),
            'latitude' => (float) ($_POST['latitude'] ?? 0),
            'longitude' => (float) ($_POST['longitude'] ?? 0)
        ];

        $selectedMaterials = $_POST['materials'] ?? [];

        $errors = $this->locationModel->validateData($data);

        if (empty($errors)) {
            try {
                $locationId = $this->locationModel->create($data);

                // Add materials (always in stock, no quantity needed)
                if (!empty($selectedMaterials)) {
                    $this->addMaterialsToLocation($locationId, $selectedMaterials);
                }

                $this->logActivity('location_created', "Created location: {$data['name']}");
                $this->redirect('page=locations&success=' . urlencode('Location added successfully'));
            } catch (Exception $e) {
                $this->redirect('page=locations&error=' . urlencode('Failed to create location: ' . $e->getMessage()));
            }
        } else {
            $errorMsg = implode(', ', $errors);
            $this->redirect('page=locations&error=' . urlencode($errorMsg));
        }
    }

    private function handleEdit($id) {
        $location = $this->locationModel->find($id);

        if (!$location) {
            $this->redirect('page=locations&error=' . urlencode('Location not found'));
            return;
        }

        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'address_line1' => trim($_POST['address_line1'] ?? ''),
            'address_line2' => trim($_POST['address_line2'] ?? ''),
            'city' => trim($_POST['city'] ?? ''),
            'province' => trim($_POST['province'] ?? ''),
            'postal_code' => trim($_POST['postal_code'] ?? ''),
            'country' => trim($_POST['country'] ?? 'Canada'),
            'latitude' => (float) ($_POST['latitude'] ?? 0),
            'longitude' => (float) ($_POST['longitude'] ?? 0)
        ];

        $selectedMaterials = $_POST['materials'] ?? [];

        $errors = $this->locationModel->validateData($data);

        if (empty($errors)) {
            try {
                $this->locationModel->update($id, $data);
                $this->updateLocationMaterials($id, $selectedMaterials);

                $this->logActivity('location_updated', "Updated location: {$data['name']}");
                $this->redirect('page=locations&success=' . urlencode('Location updated successfully'));
            } catch (Exception $e) {
                $this->redirect('page=locations&error=' . urlencode('Failed to update location: ' . $e->getMessage()));
            }
        } else {
            $errorMsg = implode(', ', $errors);
            $this->redirect('page=locations&error=' . urlencode($errorMsg));
        }
    }

    private function addMaterialsToLocation($locationId, $selectedMaterials) {
        foreach ($selectedMaterials as $materialId => $materialData) {
            if (isset($materialData['selected']) && $materialData['selected']) {
                // Check if already exists
                $existing = $this->db->fetch(
                    "SELECT id FROM location_materials WHERE location_id = ? AND material_id = ?",
                    [$locationId, $materialId]
                );

                if (!$existing) {
                    $this->db->query(
                        "INSERT INTO location_materials (location_id, material_id, notes) VALUES (?, ?, ?)",
                        [$locationId, $materialId, 'materials production']
                    );
                }
            }
        }
    }

    private function updateLocationMaterials($locationId, $selectedMaterials) {
        // Remove all existing materials for this location
        $this->db->query("DELETE FROM location_materials WHERE location_id = ?", [$locationId]);

        // Add selected materials
        if (!empty($selectedMaterials)) {
            $this->addMaterialsToLocation($locationId, $selectedMaterials);
        }
    }

    public function add() {
        $errors = [];
        $data = [];
        $selectedMaterials = [];

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = [
                'name' => trim($_POST['name'] ?? ''),
                'address_line1' => trim($_POST['address_line1'] ?? ''),
                'address_line2' => trim($_POST['address_line2'] ?? ''),
                'city' => trim($_POST['city'] ?? ''),
                'province' => trim($_POST['province'] ?? ''),
                'postal_code' => trim($_POST['postal_code'] ?? ''),
                'country' => trim($_POST['country'] ?? 'Canada'),
                'latitude' => (float) ($_POST['latitude'] ?? 0),
                'longitude' => (float) ($_POST['longitude'] ?? 0)
            ];

            $selectedMaterials = $_POST['materials'] ?? [];

            $errors = $this->locationModel->validateData($data);

            if (empty($errors)) {
                try {
                    $locationId = $this->locationModel->create($data);

                    // Add materials
                    if (!empty($selectedMaterials)) {
                        $this->addMaterialsToLocation($locationId, $selectedMaterials);
                    }

                    $this->logActivity('location_created', "Created location: {$data['name']}");
                    $this->redirect('page=locations&success=' . urlencode('Location added successfully'));
                } catch (Exception $e) {
                    $errors['general'] = 'Failed to create location: ' . $e->getMessage();
                }
            }
        }

        $materials = $this->materialModel->getActiveForLocation();

        $this->view('locations/form', [
            'location' => $data,
            'materials' => $materials,
            'selectedMaterials' => $selectedMaterials,
            'errors' => $errors,
            'isEdit' => false,
            'googleMapsApiKey' => $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY)
        ]);
    }

    public function edit($id) {
        $location = $this->locationModel->find($id);

        if (!$location) {
            $this->redirect('page=locations&error=' . urlencode('Location not found'));
        }

        $errors = [];
        $selectedMaterials = [];

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = [
                'name' => trim($_POST['name'] ?? ''),
                'address_line1' => trim($_POST['address_line1'] ?? ''),
                'address_line2' => trim($_POST['address_line2'] ?? ''),
                'city' => trim($_POST['city'] ?? ''),
                'province' => trim($_POST['province'] ?? ''),
                'postal_code' => trim($_POST['postal_code'] ?? ''),
                'country' => trim($_POST['country'] ?? 'Canada'),
                'latitude' => (float) ($_POST['latitude'] ?? 0),
                'longitude' => (float) ($_POST['longitude'] ?? 0)
            ];

            $selectedMaterials = $_POST['materials'] ?? [];

            $errors = $this->locationModel->validateData($data);

            if (empty($errors)) {
                try {
                    $this->locationModel->update($id, $data);
                    $this->updateLocationMaterials($id, $selectedMaterials);

                    $this->logActivity('location_updated', "Updated location: {$data['name']}");
                    $this->redirect('page=locations&success=' . urlencode('Location updated successfully'));
                } catch (Exception $e) {
                    $errors['general'] = 'Failed to update location: ' . $e->getMessage();
                }
            } else {
                $location = array_merge($location, $data);
            }
        }

        $materials = $this->materialModel->getActiveForLocation($id);

        $this->view('locations/form', [
            'location' => $location,
            'materials' => $materials,
            'selectedMaterials' => $selectedMaterials,
            'errors' => $errors,
            'isEdit' => true,
            'googleMapsApiKey' => $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY)
        ]);
    }

    public function toggle($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['success' => false, 'message' => 'Invalid request method'], 405);
        }

        $location = $this->locationModel->find($id);

        if (!$location) {
            $this->json(['success' => false, 'message' => 'Location not found'], 404);
        }

        try {
            $this->locationModel->toggleStatus($id);
            $newStatus = $location['active'] ? 'deactivated' : 'activated';

            $this->logActivity('location_toggled', "Location {$newStatus}: {$location['name']}");

            $this->json([
                'success' => true,
                'message' => "Location {$newStatus} successfully",
                'active' => !$location['active']
            ]);
        } catch (Exception $e) {
            $this->json(['success' => false, 'message' => 'Failed to toggle location status'], 500);
        }
    }

    // Return assigned materials for a location (IDs and details)
    public function materials($id) {
        if (!$id) {
            $this->json(['success' => false, 'message' => 'Missing location id'], 400);
        }
        $assigned = $this->db->fetchAll(
            "SELECT lm.material_id, m.name, m.code, m.price FROM location_materials lm JOIN materials m ON m.id = lm.material_id WHERE lm.location_id = ?",
            [$id]
        );
        $this->json([
            'success' => true,
            'material_ids' => array_map(function($r){ return (int)$r['material_id']; }, $assigned),
            'materials' => $assigned
        ]);
    }
}
?>