<?php
class HomeController extends Controller {
    private $locationModel;
    private $materialModel;

    public function __construct() {
        parent::__construct();
        $this->locationModel = new Location();
        $this->materialModel = new Material();
    }

    public function index() {
        $materials = $this->materialModel->findAll(['active' => 1], 'name');

        $this->view('home/index', [
            'materials' => $materials,
            'googleMapsApiKey' => $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY)
        ]);
    }

    public function search() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('');
        }

        $customerAddress = $_POST['location'] ?? '';
        $materialsInput = $_POST['materials'] ?? '';

        if (empty($customerAddress)) {
            $this->redirect('?error=' . urlencode('Please enter your address'));
        }

        if (empty($materialsInput)) {
            $this->redirect('?error=' . urlencode('Please enter at least one material'));
        }

        // Parse materials from comma-separated string
        $requestedMaterials = array_map('trim', explode(',', $materialsInput));
        $requestedMaterials = array_filter($requestedMaterials, function($material) {
            return !empty($material);
        });

        if (empty($requestedMaterials)) {
            $this->redirect('?error=' . urlencode('Please enter valid materials'));
        }

        // Step 1: Geocode customer's address
        $customerCoords = $this->geocodeAddress($customerAddress);

        if (!$customerCoords) {
            $this->redirect('?error=' . urlencode('Invalid address. Please try a different address.'));
        }

        // Step 2: Find available locations that have the requested materials
        $locationsWithMaterials = $this->findCompanyLocationsWithMaterials($requestedMaterials);

        if (empty($locationsWithMaterials)) {
            // No locations found with any of the materials
            $this->view('home/results', [
                'userLocation' => $customerAddress,
                'userCoords' => $customerCoords,
                'locations' => [],
                'selectedMaterials' => $requestedMaterials,
                'materialsInput' => $materialsInput,
                'searchMessage' => 'No locations have the materials you requested.',
                'googleMapsApiKey' => $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY)
            ]);
            return;
        }

        // Step 3: Calculate distances from customer address to each location
        $locationsWithDistance = [];
        foreach ($locationsWithMaterials as $location) {
            // Calculate straight-line distance
            $distance = $this->calculateDistance(
                $customerCoords['lat'],
                $customerCoords['lng'],
                $location['latitude'],
                $location['longitude']
            );

            $location['distance'] = $distance;

            // Get driving details from customer address to company location
            $drivingData = $this->getDrivingData($customerAddress, $location);
            $location = array_merge($location, $drivingData);

            $locationsWithDistance[] = $location;
        }

        // Step 4: Sort by one-way driving time (shortest time first)
        usort($locationsWithDistance, function($a, $b) {
            // If both have duration_seconds, sort by that
            if (isset($a['duration_seconds']) && isset($b['duration_seconds'])) {
                return $a['duration_seconds'] <=> $b['duration_seconds'];
            }
            // If only one has duration_seconds, prioritize it
            if (isset($a['duration_seconds']) && !isset($b['duration_seconds'])) {
                return -1;
            }
            if (!isset($a['duration_seconds']) && isset($b['duration_seconds'])) {
                return 1;
            }
            // If neither has duration_seconds, fall back to distance
            return $a['distance'] <=> $b['distance'];
        });

        $this->view('home/results', [
            'userLocation' => $customerAddress,
            'userCoords' => $customerCoords,
            'locations' => $locationsWithDistance,
            'selectedMaterials' => $requestedMaterials,
            'materialsInput' => $materialsInput,
            'searchMessage' => 'Found ' . count($locationsWithDistance) . ' location(s) with your requested materials.',
            'googleMapsApiKey' => $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY)
        ]);
    }

    public function howToUse() {
        $this->view('home/how-to-use');
    }

    public function releases() {
        $this->view('home/releases');
    }

    private function geocodeAddress($address) {
        $apiKey = $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY);
        $url = "https://maps.googleapis.com/maps/api/geocode/json?address=" . urlencode($address) . "&key=" . $apiKey;

        $response = file_get_contents($url);
        $data = json_decode($response, true);

        if ($data['status'] === 'OK' && !empty($data['results'])) {
            $result = $data['results'][0];
            return [
                'lat' => $result['geometry']['location']['lat'],
                'lng' => $result['geometry']['location']['lng'],
                'formatted_address' => $result['formatted_address']
            ];
        }

        return false;
    }

    private function calculateDistance($lat1, $lng1, $lat2, $lng2) {
        $earthRadius = 6371; // Earth's radius in kilometers

        $dLat = deg2rad($lat2 - $lat1);
        $dLng = deg2rad($lng2 - $lng1);

        $a = sin($dLat/2) * sin($dLat/2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($dLng/2) * sin($dLng/2);

        $c = 2 * atan2(sqrt($a), sqrt(1-$a));

        return $earthRadius * $c;
    }

    private function getDrivingData($customerAddress, $companyLocation) {
        $companyAddress = $companyLocation['address_line1'] . ', ' . $companyLocation['city'] . ', ' . $companyLocation['province'];

        // Check cache first (from customer address to company location)
        // Reduce cache time to 2 hours for more accurate traffic data
        $cached = $this->db->fetch(
            "SELECT * FROM distance_cache 
             WHERE origin_address = ? AND destination_address = ? 
             AND expires_at > NOW()",
            [$customerAddress, $companyAddress]
        );

        if ($cached) {
            return [
                'driving_distance' => $cached['driving_distance_text'],
                'driving_duration' => $cached['driving_duration_text'],
                'calculated_duration' => $this->formatDuration($cached['calculated_duration_minutes']),
                'duration_seconds' => $cached['duration_seconds'],
                'traffic_duration_seconds' => isset($cached['traffic_duration_seconds']) ? $cached['traffic_duration_seconds'] : $cached['duration_seconds'],
                'traffic_duration' => isset($cached['traffic_duration_text']) ? $cached['traffic_duration_text'] : $cached['driving_duration_text']
            ];
        }

        // Get from Google Maps API with traffic-aware parameters
        $apiKey = $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY);
        
        // Use current time + 5 minutes for departure to get real-time traffic
        $departureTime = time() + 300; // 5 minutes from now
        
        $url = "https://maps.googleapis.com/maps/api/distancematrix/json?" . http_build_query([
            'origins' => $customerAddress,
            'destinations' => $companyAddress,
            'units' => 'metric',
            'departure_time' => $departureTime,
            'traffic_model' => 'best_guess', // Options: best_guess, pessimistic, optimistic
            'key' => $apiKey
        ]);

        $response = file_get_contents($url);
        $data = json_decode($response, true);

        if ($data['status'] === 'OK' &&
            !empty($data['rows'][0]['elements'][0]) &&
            $data['rows'][0]['elements'][0]['status'] === 'OK') {

            $element = $data['rows'][0]['elements'][0];
            
            // Get base duration (without traffic)
            $durationSeconds = $element['duration']['value'];
            $durationMinutes = round($durationSeconds / 60);
            
            // Get traffic-aware duration if available
            $trafficDurationSeconds = $durationSeconds;
            $trafficDurationText = $element['duration']['text'];
            
            if (isset($element['duration_in_traffic'])) {
                $trafficDurationSeconds = $element['duration_in_traffic']['value'];
                $trafficDurationText = $element['duration_in_traffic']['text'];
            }
            
            $trafficDurationMinutes = round($trafficDurationSeconds / 60);

            // Calculate custom duration based on traffic-aware time: ((traffic duration x 2) + 25%) + 20 min
            $calculatedMinutes = (($trafficDurationMinutes * 2) * 1.25) + 20;

            // Cache the result for 2 hours (shorter cache for better traffic accuracy)
            $this->db->query(
                "INSERT INTO distance_cache 
                 (origin_address, destination_address, destination_lat, destination_lng, 
                  distance_km, duration_seconds, driving_distance_text, driving_duration_text, 
                  traffic_duration_seconds, traffic_duration_text, calculated_duration_minutes, expires_at) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 2 HOUR))
                 ON DUPLICATE KEY UPDATE
                 distance_km = VALUES(distance_km),
                 duration_seconds = VALUES(duration_seconds),
                 driving_distance_text = VALUES(driving_distance_text),
                 driving_duration_text = VALUES(driving_duration_text),
                 traffic_duration_seconds = VALUES(traffic_duration_seconds),
                 traffic_duration_text = VALUES(traffic_duration_text),
                 calculated_duration_minutes = VALUES(calculated_duration_minutes),
                 expires_at = VALUES(expires_at)",
                [
                    $customerAddress,
                    $companyAddress,
                    $companyLocation['latitude'],
                    $companyLocation['longitude'],
                    round($element['distance']['value'] / 1000, 2),
                    $durationSeconds,
                    $element['distance']['text'],
                    $element['duration']['text'],
                    $trafficDurationSeconds,
                    $trafficDurationText,
                    (int) round($calculatedMinutes)
                ]
            );

            return [
                'driving_distance' => $element['distance']['text'],
                'driving_duration' => $trafficDurationText, // Use traffic-aware duration
                'calculated_duration' => $this->formatDuration($calculatedMinutes),
                'duration_seconds' => $trafficDurationSeconds, // Use traffic-aware duration for sorting
                'traffic_duration_seconds' => $trafficDurationSeconds,
                'traffic_duration' => $trafficDurationText
            ];
        }

        return [
            'driving_distance' => 'N/A',
            'driving_duration' => 'N/A',
            'calculated_duration' => 'N/A'
        ];
    }

    private function formatDuration($minutes) {
        // Human readable: X hours Y minutes
        $totalMinutes = (int) round((float) $minutes);
        $hours = intdiv($totalMinutes, 60);
        $mins = $totalMinutes % 60;
        $parts = [];
        if ($hours > 0) $parts[] = $hours . ' hour' . ($hours === 1 ? '' : 's');
        if ($mins > 0) $parts[] = $mins . ' minute' . ($mins === 1 ? '' : 's');
        if (empty($parts)) return '0 minutes';
        return implode(' ', $parts);
    }

    /**
     * Find locations that have the materials requested by the customer
     * This is the core business logic: customer wants materials, we find our locations that have them
     */
    private function findCompanyLocationsWithMaterials($requestedMaterialNames) {
        if (empty($requestedMaterialNames)) {
            return [];
        }

        // Step 1: Find material IDs that match the customer's requests (fuzzy matching)
        $matchedMaterialIds = [];
        $matchedMaterials = [];

        foreach ($requestedMaterialNames as $requestedName) {
            $requestedName = trim($requestedName);
            if (empty($requestedName)) continue;

            // Try multiple search strategies to find matching materials
            $foundMaterial = $this->findBestMaterialMatch($requestedName);

            if ($foundMaterial) {
                $matchedMaterialIds[] = $foundMaterial['id'];
                $matchedMaterials[] = $foundMaterial;
            }
        }

        if (empty($matchedMaterialIds)) {
            return []; // No materials found matching customer requests
        }

        // Remove duplicates
        $matchedMaterialIds = array_unique($matchedMaterialIds);

        // Step 2: Find locations that have ANY of these materials
        $placeholders = str_repeat('?,', count($matchedMaterialIds) - 1) . '?';

        $sql = "
            SELECT DISTINCT l.*, 
                   COUNT(DISTINCT lm.material_id) as available_material_count,
                   GROUP_CONCAT(DISTINCT m.name ORDER BY m.name SEPARATOR ', ') as available_material_names
            FROM locations l
            INNER JOIN location_materials lm ON l.id = lm.location_id
            INNER JOIN materials m ON lm.material_id = m.id
            WHERE l.active = 1 
            AND lm.material_id IN ($placeholders)
            GROUP BY l.id
            ORDER BY available_material_count DESC, l.name
        ";

        $companyLocations = $this->db->fetchAll($sql, $matchedMaterialIds);

        // Step 3: Add detailed material information for each location
        foreach ($companyLocations as &$location) {
            // Get all requested materials that this location has (updated to remove quantity/unit)
            $location['available_materials'] = $this->db->fetchAll(
                "SELECT m.id, m.name, m.code, m.price, lm.notes
                 FROM materials m 
                 JOIN location_materials lm ON m.id = lm.material_id 
                 WHERE lm.location_id = ? AND m.id IN ($placeholders)
                 ORDER BY m.name",
                array_merge([$location['id']], $matchedMaterialIds)
            );

            // Add metadata about the search
            $location['matched_materials'] = $matchedMaterials;
            $location['customer_requests'] = $requestedMaterialNames;
            $location['match_percentage'] = (int) round(($location['available_material_count'] / count($requestedMaterialNames)) * 100);
        }

        return $companyLocations;
    }

    /**
     * Find the best matching material for a customer's request
     */
    private function findBestMaterialMatch($requestedName) {
        // Strategy 1: Exact match
        $exactMatch = $this->db->fetch(
            "SELECT id, name, code FROM materials WHERE name = ? AND active = 1",
            [$requestedName]
        );
        if ($exactMatch) return $exactMatch;

        // Strategy 2: Case-insensitive exact match
        $caseMatch = $this->db->fetch(
            "SELECT id, name, code FROM materials WHERE LOWER(name) = LOWER(?) AND active = 1",
            [$requestedName]
        );
        if ($caseMatch) return $caseMatch;

        // Strategy 3: Contains search (prioritize starts with, then contains)
        $containsMatch = $this->db->fetch(
            "SELECT id, name, code FROM materials 
             WHERE name LIKE ? AND active = 1 
             ORDER BY 
                CASE 
                    WHEN name LIKE ? THEN 1
                    WHEN name LIKE ? THEN 2
                    ELSE 3
                END
             LIMIT 1",
            ['%' . $requestedName . '%', $requestedName . '%', '%' . $requestedName]
        );
        if ($containsMatch) return $containsMatch;

        // Strategy 4: Word-based search (all words must be found)
        $words = explode(' ', $requestedName);
        $significantWords = array_filter($words, function($word) {
            return strlen(trim($word)) >= 3; // Only words with 3+ characters
        });

        if (!empty($significantWords)) {
            $wordConditions = array_map(function($word) {
                return "name LIKE ?";
            }, $significantWords);

            $wordParams = array_map(function($word) {
                return '%' . trim($word) . '%';
            }, $significantWords);

            $wordMatch = $this->db->fetch(
                "SELECT id, name, code FROM materials 
                 WHERE (" . implode(' AND ', $wordConditions) . ") AND active = 1 
                 LIMIT 1",
                $wordParams
            );
            if ($wordMatch) return $wordMatch;
        }

        // Strategy 5: Partial word matching (any word matches)
        if (!empty($significantWords)) {
            $orConditions = array_map(function($word) {
                return "name LIKE ?";
            }, $significantWords);

            $orParams = array_map(function($word) {
                return '%' . trim($word) . '%';
            }, $significantWords);

            $partialMatch = $this->db->fetch(
                "SELECT id, name, code FROM materials 
                 WHERE (" . implode(' OR ', $orConditions) . ") AND active = 1 
                 ORDER BY name
                 LIMIT 1",
                $orParams
            );
            if ($partialMatch) return $partialMatch;
        }

        return null; // No match found
    }
}
?>