<?php
class AuthController extends Controller {
    private $userModel;
    
    public function __construct() {
        parent::__construct();
        $this->userModel = new User();
    }
    
    public function login() {
        // Redirect if already logged in
        if (isset($_SESSION['user_id'])) {
            $this->redirect('page=materials');
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $username = trim($_POST['username'] ?? '');
            $password = $_POST['password'] ?? '';
            
            if (empty($username) || empty($password)) {
                $error = 'Username and password are required';
            } else {
                // Check for too many failed attempts
                $failedAttempts = $this->userModel->getRecentFailedAttempts($username);
                
                if ($failedAttempts >= MAX_LOGIN_ATTEMPTS) {
                    $error = 'Too many failed login attempts. Please try again later.';
                } else {
                    $user = $this->userModel->authenticate($username, $password);
                    
                    if ($user) {
                        // Successful login
                        // Support LDAP-only users without a local DB record
                        $_SESSION['user_id'] = isset($user['id']) && $user['id'] !== null && $user['id'] !== ''
                            ? $user['id']
                            : ('ldap:' . $user['username']);
                        $_SESSION['username'] = $user['username'];
                        $_SESSION['display_name'] = $user['display_name'];
                        $_SESSION['auth_method'] = $user['auth_method'];
                        $_SESSION['last_activity'] = time();
                        
                        // Update last login only if we have a local numeric ID
                        if (isset($user['id']) && is_numeric($user['id'])) {
                            $this->userModel->updateLastLogin((int)$user['id']);
                        }
                        
                        // Log successful attempt
                        $this->userModel->logLoginAttempt(
                            $username, 
                            true, 
                            $_SERVER['REMOTE_ADDR'] ?? null,
                            $_SERVER['HTTP_USER_AGENT'] ?? null
                        );
                        
                        $this->logActivity('login', 'User logged in successfully');
                        
                        $this->redirect('page=materials');
                    } else {
                        // Failed login
                        $error = 'Invalid username or password';
                        
                        $this->userModel->logLoginAttempt(
                            $username, 
                            false, 
                            $_SERVER['REMOTE_ADDR'] ?? null,
                            $_SERVER['HTTP_USER_AGENT'] ?? null
                        );
                    }
                }
            }
        }
        
        // Check for timeout message
        if (isset($_GET['timeout'])) {
            $error = 'Your session has expired. Please log in again.';
        }
        
        $this->view('auth/login', [
            'error' => $error,
            'ldapEnabled' => $this->getSetting('ldap_enabled', false)
        ]);
    }
    
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            $this->logActivity('logout', 'User logged out');
        }
        
        session_destroy();
        $this->redirect('page=login&logged_out=1');
    }
}
?>