<?php
class ApiController extends Controller {
    private $locationModel;
    private $materialModel;
    
    public function __construct() {
        parent::__construct();
        $this->locationModel = new Location();
        $this->materialModel = new Material();
    }
    
    public function distance() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['success' => false, 'message' => 'Invalid request method'], 405);
        }
        
        $origin = $_POST['origin'] ?? '';
        $destination = $_POST['destination'] ?? '';
        
        if (empty($origin) || empty($destination)) {
            $this->json(['success' => false, 'message' => 'Origin and destination are required'], 400);
        }
        
        // Check cache first
        $cached = $this->db->fetch(
            "SELECT * FROM distance_cache 
             WHERE origin_address = ? AND destination_address = ? 
             AND expires_at > NOW()",
            [$origin, $destination]
        );
        
        if ($cached) {
            $this->json([
                'success' => true,
                'distance' => $cached['driving_distance_text'],
                'duration' => isset($cached['traffic_duration_text']) ? $cached['traffic_duration_text'] : $cached['driving_duration_text'],
                'calculated_duration' => $this->formatDuration($cached['calculated_duration_minutes']),
                'traffic_duration' => isset($cached['traffic_duration_text']) ? $cached['traffic_duration_text'] : $cached['driving_duration_text'],
                'base_duration' => $cached['driving_duration_text']
            ]);
        }
        
        // Get from Google Maps API with traffic-aware parameters
        $apiKey = $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY);
        
        // Use current time + 5 minutes for departure to get real-time traffic
        $departureTime = time() + 300; // 5 minutes from now
        
        $url = "https://maps.googleapis.com/maps/api/distancematrix/json?" . http_build_query([
            'origins' => $origin,
            'destinations' => $destination,
            'units' => 'metric',
            'departure_time' => $departureTime,
            'traffic_model' => 'best_guess', // Options: best_guess, pessimistic, optimistic
            'key' => $apiKey
        ]);
        
        $response = file_get_contents($url);
        $data = json_decode($response, true);
        
        if ($data['status'] === 'OK' && 
            !empty($data['rows'][0]['elements'][0]) && 
            $data['rows'][0]['elements'][0]['status'] === 'OK') {
            
            $element = $data['rows'][0]['elements'][0];
            
            // Get base duration (without traffic)
            $durationSeconds = $element['duration']['value'];
            $durationMinutes = round($durationSeconds / 60);
            
            // Get traffic-aware duration if available
            $trafficDurationSeconds = $durationSeconds;
            $trafficDurationText = $element['duration']['text'];
            
            if (isset($element['duration_in_traffic'])) {
                $trafficDurationSeconds = $element['duration_in_traffic']['value'];
                $trafficDurationText = $element['duration_in_traffic']['text'];
            }
            
            $trafficDurationMinutes = round($trafficDurationSeconds / 60);
            
            // Calculate custom duration based on traffic-aware time: ((traffic duration x 2) + 25%) + 20 min
            $calculatedMinutes = (($trafficDurationMinutes * 2) * 1.25) + 20;
            
            // Cache the result for 2 hours (shorter cache for better traffic accuracy)
            $this->db->query(
                "INSERT INTO distance_cache 
                 (origin_address, destination_address, destination_lat, destination_lng, 
                  distance_km, duration_seconds, driving_distance_text, driving_duration_text, 
                  traffic_duration_seconds, traffic_duration_text, calculated_duration_minutes, expires_at) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 2 HOUR))
                 ON DUPLICATE KEY UPDATE
                 distance_km = VALUES(distance_km),
                 duration_seconds = VALUES(duration_seconds),
                 driving_distance_text = VALUES(driving_distance_text),
                 driving_duration_text = VALUES(driving_duration_text),
                 traffic_duration_seconds = VALUES(traffic_duration_seconds),
                 traffic_duration_text = VALUES(traffic_duration_text),
                 calculated_duration_minutes = VALUES(calculated_duration_minutes),
                 expires_at = VALUES(expires_at)",
                [
                    $origin,
                    $destination,
                    0, // We don't have destination coordinates in this context
                    0,
                    round($element['distance']['value'] / 1000, 2),
                    $durationSeconds,
                    $element['distance']['text'],
                    $element['duration']['text'],
                    $trafficDurationSeconds,
                    $trafficDurationText,
                    (int) round($calculatedMinutes)
                ]
            );
            
            $this->json([
                'success' => true,
                'distance' => $element['distance']['text'],
                'duration' => $trafficDurationText, // Use traffic-aware duration
                'calculated_duration' => $this->formatDuration($calculatedMinutes),
                'traffic_duration' => $trafficDurationText,
                'base_duration' => $element['duration']['text']
            ]);
        } else {
            $this->json([
                'success' => false,
                'message' => 'Unable to calculate distance',
                'debug' => $data
            ], 400);
        }
    }
    
    public function geocode() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json(['success' => false, 'message' => 'Invalid request method'], 405);
        }
        
        $address = $_POST['address'] ?? '';
        
        if (empty($address)) {
            $this->json(['success' => false, 'message' => 'Address is required'], 400);
        }
        
        $apiKey = $this->getSetting('google_maps_api_key', GOOGLE_MAPS_API_KEY);
        $url = "https://maps.googleapis.com/maps/api/geocode/json?address=" . urlencode($address) . "&key=" . $apiKey;
        
        $response = file_get_contents($url);
        $data = json_decode($response, true);
        
        if ($data['status'] === 'OK' && !empty($data['results'])) {
            $result = $data['results'][0];
            $this->json([
                'success' => true,
                'lat' => $result['geometry']['location']['lat'],
                'lng' => $result['geometry']['location']['lng'],
                'formatted_address' => $result['formatted_address']
            ]);
        } else {
            $this->json([
                'success' => false,
                'message' => 'Unable to geocode address',
                'debug' => $data
            ], 400);
        }
    }
    
    public function materials() {
        $search = $_GET['search'] ?? '';
        
        if (empty($search)) {
            $materials = $this->materialModel->findAll(['active' => 1], 'name', 20);
        } else {
            $materials = $this->db->fetchAll(
                "SELECT id, name, code, material_group 
                 FROM materials 
                 WHERE active = 1 AND (name LIKE ? OR code LIKE ?) 
                 ORDER BY name 
                 LIMIT 20",
                ['%' . $search . '%', '%' . $search . '%']
            );
        }
        
        $this->json([
            'success' => true,
            'materials' => $materials
        ]);
    }
    
    private function formatDuration($minutes) {
        // Human readable: X hours Y minutes
        $totalMinutes = (float) $minutes;
        $hours = (int) floor($totalMinutes / 60);
        $mins = (int) ($totalMinutes % 60);
        $parts = [];
        if ($hours > 0) $parts[] = $hours . ' hour' . ($hours === 1 ? '' : 's');
        if ($mins > 0) $parts[] = $mins . ' minute' . ($mins === 1 ? '' : 's');
        if (empty($parts)) return '0 minutes';
        return implode(' ', $parts);
    }
}
?>