<?php
/**
 * Fix Duplicate Inventory Records (Simple Version)
 * 
 * This script consolidates duplicate inventory records without requiring
 * the full application bootstrap.
 * 
 * Usage: php fix_inventory_duplicates_simple.php
 */

// Database credentials - update these if needed
$host = 'localhost';
$dbname = 'handsdb';  // Update this to your database name
$username = 'root';    // Update this to your database username
$password = '';        // Update this to your database password

echo "=== Fix Duplicate Inventory Records ===\n\n";

try {
    // Connect to database
    $dsn = "mysql:host={$host};dbname={$dbname};charset=utf8mb4";
    $pdo = new PDO($dsn, $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);

    echo "✓ Connected to database successfully.\n\n";

    // Step 1: Check for duplicates
    echo "Step 1: Checking for duplicate inventory records...\n";
    $stmt = $pdo->query("
        SELECT product_id, location_id, COUNT(*) as count, SUM(quantity) as total_qty
        FROM inventory
        GROUP BY product_id, location_id
        HAVING COUNT(*) > 1
    ");
    $duplicates = $stmt->fetchAll();

    if (empty($duplicates)) {
        echo "✓ No duplicate records found. Database is clean!\n";
        exit(0);
    }

    echo "Found " . count($duplicates) . " product-location combinations with duplicates:\n";
    
    // Get product and location names for better display
    foreach ($duplicates as $dup) {
        $stmt = $pdo->prepare("
            SELECT p.part_number, p.name as product_name, l.name as location_name
            FROM products p, locations l
            WHERE p.id = :product_id AND l.id = :location_id
        ");
        $stmt->execute([
            'product_id' => $dup['product_id'],
            'location_id' => $dup['location_id']
        ]);
        $info = $stmt->fetch();
        
        echo "  - {$info['part_number']} ({$info['product_name']}) @ {$info['location_name']}: ";
        echo "{$dup['count']} records, Total quantity: {$dup['total_qty']}\n";
    }
    echo "\n";

    // Ask for confirmation
    echo "Do you want to consolidate these duplicate records? (yes/no): ";
    $handle = fopen("php://stdin", "r");
    $line = trim(fgets($handle));
    fclose($handle);

    if (strtolower($line) !== 'yes') {
        echo "Operation cancelled.\n";
        exit(0);
    }

    // Step 2: Consolidate duplicates
    echo "\nStep 2: Consolidating duplicate records...\n";
    $pdo->beginTransaction();

    try {
        // Get all inventory records grouped by product_id and location_id
        $stmt = $pdo->query("
            SELECT 
                product_id,
                location_id,
                SUM(quantity) as total_quantity,
                MAX(min_quantity) as min_quantity,
                MAX(last_updated) as last_updated,
                GROUP_CONCAT(DISTINCT remarks SEPARATOR '; ') as combined_remarks
            FROM inventory
            GROUP BY product_id, location_id
        ");
        $consolidated = $stmt->fetchAll();

        // Delete all existing inventory records
        $pdo->exec("DELETE FROM inventory");
        echo "  ✓ Deleted all existing inventory records.\n";

        // Insert consolidated records
        $insertStmt = $pdo->prepare("
            INSERT INTO inventory (product_id, location_id, quantity, min_quantity, last_updated, remarks)
            VALUES (:product_id, :location_id, :quantity, :min_quantity, :last_updated, :remarks)
        ");

        $insertedCount = 0;
        foreach ($consolidated as $record) {
            $insertStmt->execute([
                'product_id' => $record['product_id'],
                'location_id' => $record['location_id'],
                'quantity' => $record['total_quantity'],
                'min_quantity' => $record['min_quantity'],
                'last_updated' => $record['last_updated'],
                'remarks' => $record['combined_remarks'] ?: null
            ]);
            $insertedCount++;
        }

        echo "  ✓ Inserted {$insertedCount} consolidated records.\n";

        // Step 3: Ensure unique constraint exists
        echo "\nStep 3: Ensuring unique constraint exists...\n";
        
        // Check if constraint exists
        $stmt = $pdo->query("
            SELECT COUNT(*) as count
            FROM information_schema.statistics
            WHERE table_schema = DATABASE()
            AND table_name = 'inventory'
            AND index_name = 'unique_product_location'
        ");
        $constraintExists = $stmt->fetch()['count'] > 0;

        if ($constraintExists) {
            echo "  ✓ Unique constraint already exists.\n";
        } else {
            echo "  Adding unique constraint...\n";
            $pdo->exec("ALTER TABLE inventory ADD UNIQUE KEY `unique_product_location` (`product_id`, `location_id`)");
            echo "  ✓ Unique constraint added successfully.\n";
        }

        $pdo->commit();
        echo "\n✓ All duplicates have been consolidated successfully!\n\n";

        // Step 4: Show final inventory
        echo "Final inventory summary:\n";
        echo str_repeat("-", 80) . "\n";
        $stmt = $pdo->query("
            SELECT 
                p.part_number,
                p.name as product_name,
                l.name as location_name,
                i.quantity,
                i.min_quantity
            FROM inventory i
            JOIN products p ON i.product_id = p.id
            JOIN locations l ON i.location_id = l.id
            ORDER BY p.part_number, l.name
        ");
        $inventory = $stmt->fetchAll();

        foreach ($inventory as $item) {
            printf("%-20s %-30s @ %-15s: %5d units (min: %d)\n",
                $item['part_number'],
                substr($item['product_name'], 0, 30),
                $item['location_name'],
                $item['quantity'],
                $item['min_quantity']
            );
        }
        echo str_repeat("-", 80) . "\n";

    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }

} catch (PDOException $e) {
    echo "\n✗ Database error: " . $e->getMessage() . "\n";
    exit(1);
} catch (Exception $e) {
    echo "\n✗ Error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n✓ Done! Your inventory is now clean and duplicate-free.\n";
