<?php
/**
 * Fix Duplicate Inventory Records
 * 
 * This script consolidates duplicate inventory records that were created
 * when the size_id column was removed from the inventory table.
 * 
 * Run this script once from the command line:
 * php fix_inventory_duplicates.php
 */

require_once __DIR__ . '/config/database.php';

try {
    // Connect to database
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
    $pdo = new PDO($dsn, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);

    echo "Connected to database successfully.\n\n";

    // Step 1: Check for duplicates
    echo "Step 1: Checking for duplicate inventory records...\n";
    $stmt = $pdo->query("
        SELECT product_id, location_id, COUNT(*) as count, SUM(quantity) as total_qty
        FROM inventory
        GROUP BY product_id, location_id
        HAVING COUNT(*) > 1
    ");
    $duplicates = $stmt->fetchAll();

    if (empty($duplicates)) {
        echo "No duplicate records found. Database is clean!\n";
        exit(0);
    }

    echo "Found " . count($duplicates) . " product-location combinations with duplicates:\n";
    foreach ($duplicates as $dup) {
        echo "  - Product ID {$dup['product_id']}, Location ID {$dup['location_id']}: ";
        echo "{$dup['count']} records, Total quantity: {$dup['total_qty']}\n";
    }
    echo "\n";

    // Step 2: Consolidate duplicates
    echo "Step 2: Consolidating duplicate records...\n";
    $pdo->beginTransaction();

    try {
        // Get all inventory records grouped by product_id and location_id
        $stmt = $pdo->query("
            SELECT 
                product_id,
                location_id,
                SUM(quantity) as total_quantity,
                MAX(min_quantity) as min_quantity,
                MAX(last_updated) as last_updated,
                GROUP_CONCAT(DISTINCT remarks SEPARATOR '; ') as combined_remarks
            FROM inventory
            GROUP BY product_id, location_id
        ");
        $consolidated = $stmt->fetchAll();

        // Delete all existing inventory records
        $pdo->exec("DELETE FROM inventory");
        echo "  Deleted all existing inventory records.\n";

        // Insert consolidated records
        $insertStmt = $pdo->prepare("
            INSERT INTO inventory (product_id, location_id, quantity, min_quantity, last_updated, remarks)
            VALUES (:product_id, :location_id, :quantity, :min_quantity, :last_updated, :remarks)
        ");

        $insertedCount = 0;
        foreach ($consolidated as $record) {
            $insertStmt->execute([
                'product_id' => $record['product_id'],
                'location_id' => $record['location_id'],
                'quantity' => $record['total_quantity'],
                'min_quantity' => $record['min_quantity'],
                'last_updated' => $record['last_updated'],
                'remarks' => $record['combined_remarks'] ?: null
            ]);
            $insertedCount++;
        }

        echo "  Inserted {$insertedCount} consolidated records.\n";

        // Step 3: Ensure unique constraint exists
        echo "\nStep 3: Ensuring unique constraint exists...\n";
        
        // Check if constraint exists
        $stmt = $pdo->query("
            SELECT COUNT(*) as count
            FROM information_schema.statistics
            WHERE table_schema = DATABASE()
            AND table_name = 'inventory'
            AND index_name = 'unique_product_location'
        ");
        $constraintExists = $stmt->fetch()['count'] > 0;

        if ($constraintExists) {
            echo "  Unique constraint already exists.\n";
        } else {
            echo "  Adding unique constraint...\n";
            $pdo->exec("ALTER TABLE inventory ADD UNIQUE KEY `unique_product_location` (`product_id`, `location_id`)");
            echo "  Unique constraint added successfully.\n";
        }

        $pdo->commit();
        echo "\n✓ All duplicates have been consolidated successfully!\n\n";

        // Step 4: Show final inventory
        echo "Final inventory summary:\n";
        $stmt = $pdo->query("
            SELECT 
                p.part_number,
                p.name as product_name,
                l.name as location_name,
                i.quantity,
                i.min_quantity
            FROM inventory i
            JOIN products p ON i.product_id = p.id
            JOIN locations l ON i.location_id = l.id
            ORDER BY p.part_number, l.name
        ");
        $inventory = $stmt->fetchAll();

        foreach ($inventory as $item) {
            echo "  {$item['part_number']} - {$item['product_name']} @ {$item['location_name']}: ";
            echo "{$item['quantity']} units (min: {$item['min_quantity']})\n";
        }

    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }

} catch (PDOException $e) {
    echo "Database error: " . $e->getMessage() . "\n";
    exit(1);
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\nDone!\n";
