<?php
/**
 * Export Current Database with Duplicate Fix
 * 
 * This script exports all data from the current database and generates
 * a complete SQL file that can be used to create a clean database.
 * 
 * Usage: php export_database_with_fix.php > complete_export.sql
 */

// Database credentials - update these if needed
$host = 'localhost';
$dbname = 'handsdb';  // Your current database name
$username = 'root';
$password = '';

try {
    $pdo = new PDO("mysql:host={$host};dbname={$dbname};charset=utf8mb4", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);

    echo "-- ============================================================================\n";
    echo "-- Complete Database Export with Duplicate Inventory Fix\n";
    echo "-- Generated: " . date('Y-m-d H:i:s') . "\n";
    echo "-- Source Database: {$dbname}\n";
    echo "-- ============================================================================\n";
    echo "-- Usage:\n";
    echo "--   1. Create new database: CREATE DATABASE handsdb_clean;\n";
    echo "--   2. Import this file: mysql -u root -p handsdb_clean < complete_export.sql\n";
    echo "-- ============================================================================\n\n";

    echo "SET NAMES utf8mb4;\n";
    echo "SET FOREIGN_KEY_CHECKS = 0;\n";
    echo "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n";
    echo "SET time_zone = \"+00:00\";\n\n";

    // Read the schema from the template file
    $schemaFile = __DIR__ . '/database/schema.sql';
    if (file_exists($schemaFile)) {
        echo "-- ============================================================================\n";
        echo "-- TABLE STRUCTURES\n";
        echo "-- ============================================================================\n\n";
        echo file_get_contents($schemaFile);
        echo "\n\n";
    }

    echo "-- ============================================================================\n";
    echo "-- DATA EXPORT\n";
    echo "-- ============================================================================\n\n";

    // Tables to export in order (respecting foreign keys)
    $tables = [
        'categories',
        'locations',
        'departments',
        'users',
        'products',
        'employees',
        'toast_templates',
        'humor_lines',
        'stock_transfers',
        'stock_transactions',
        'activity_logs',
        'notifications',
        'user_sessions'
    ];

    foreach ($tables as $table) {
        // Check if table exists
        $stmt = $pdo->query("SHOW TABLES LIKE '{$table}'");
        if ($stmt->rowCount() == 0) {
            echo "-- Table {$table} does not exist, skipping...\n\n";
            continue;
        }

        echo "-- Data for table: {$table}\n";
        
        $stmt = $pdo->query("SELECT * FROM `{$table}`");
        $rows = $stmt->fetchAll();
        
        if (empty($rows)) {
            echo "-- No data in {$table}\n\n";
            continue;
        }

        // Get column names
        $columns = array_keys($rows[0]);
        $columnList = '`' . implode('`, `', $columns) . '`';

        echo "INSERT INTO `{$table}` ({$columnList}) VALUES\n";
        
        $values = [];
        foreach ($rows as $row) {
            $rowValues = [];
            foreach ($row as $value) {
                if ($value === null) {
                    $rowValues[] = 'NULL';
                } elseif (is_numeric($value)) {
                    $rowValues[] = $value;
                } else {
                    $rowValues[] = "'" . $pdo->quote($value) . "'";
                }
            }
            $values[] = '(' . implode(', ', $rowValues) . ')';
        }
        
        echo implode(",\n", $values) . ";\n\n";
    }

    // Special handling for inventory with duplicate consolidation
    echo "-- ============================================================================\n";
    echo "-- INVENTORY DATA WITH DUPLICATE CONSOLIDATION\n";
    echo "-- ============================================================================\n\n";

    // Check for duplicates
    $stmt = $pdo->query("
        SELECT product_id, location_id, COUNT(*) as count
        FROM inventory
        GROUP BY product_id, location_id
        HAVING COUNT(*) > 1
    ");
    $duplicates = $stmt->fetchAll();

    if (!empty($duplicates)) {
        echo "-- WARNING: Found " . count($duplicates) . " duplicate product-location combinations\n";
        echo "-- These will be consolidated by summing quantities\n\n";
    }

    // Export consolidated inventory
    $stmt = $pdo->query("
        SELECT 
            product_id,
            location_id,
            SUM(quantity) as quantity,
            MAX(min_quantity) as min_quantity,
            MAX(last_updated) as last_updated,
            GROUP_CONCAT(DISTINCT remarks SEPARATOR '; ') as remarks
        FROM inventory
        GROUP BY product_id, location_id
    ");
    $inventoryRows = $stmt->fetchAll();

    if (!empty($inventoryRows)) {
        echo "INSERT INTO `inventory` (`product_id`, `location_id`, `quantity`, `min_quantity`, `last_updated`, `remarks`) VALUES\n";
        
        $values = [];
        foreach ($inventoryRows as $row) {
            $remarks = $row['remarks'] ? "'" . addslashes($row['remarks']) . "'" : 'NULL';
            $values[] = sprintf(
                "(%d, %d, %d, %d, '%s', %s)",
                $row['product_id'],
                $row['location_id'],
                $row['quantity'],
                $row['min_quantity'],
                $row['last_updated'],
                $remarks
            );
        }
        
        echo implode(",\n", $values) . ";\n\n";
    } else {
        echo "-- No inventory data\n\n";
    }

    echo "-- ============================================================================\n";
    echo "-- VERIFICATION QUERIES\n";
    echo "-- ============================================================================\n\n";

    echo "-- Check for duplicates (should return 0)\n";
    echo "SELECT \n";
    echo "    'Inventory Duplicate Check' as check_name,\n";
    echo "    COUNT(*) as total_records,\n";
    echo "    COUNT(DISTINCT CONCAT(product_id, '-', location_id)) as unique_combinations,\n";
    echo "    COUNT(*) - COUNT(DISTINCT CONCAT(product_id, '-', location_id)) as duplicates\n";
    echo "FROM inventory;\n\n";

    echo "-- Show inventory summary\n";
    echo "SELECT \n";
    echo "    p.part_number,\n";
    echo "    p.name as product_name,\n";
    echo "    l.name as location_name,\n";
    echo "    i.quantity,\n";
    echo "    i.min_quantity\n";
    echo "FROM inventory i\n";
    echo "JOIN products p ON i.product_id = p.id\n";
    echo "JOIN locations l ON i.location_id = l.id\n";
    echo "ORDER BY p.part_number, l.name;\n\n";

    echo "SET FOREIGN_KEY_CHECKS = 1;\n\n";
    echo "-- ============================================================================\n";
    echo "-- EXPORT COMPLETE\n";
    echo "-- ============================================================================\n";

    // Output summary to stderr so it doesn't go into the SQL file
    fwrite(STDERR, "\n✓ Export completed successfully!\n");
    fwrite(STDERR, "  Tables exported: " . count($tables) . "\n");
    fwrite(STDERR, "  Inventory records: " . count($inventoryRows) . " (consolidated)\n");
    if (!empty($duplicates)) {
        fwrite(STDERR, "  Duplicates found and consolidated: " . count($duplicates) . "\n");
    }
    fwrite(STDERR, "\nTo create a new clean database:\n");
    fwrite(STDERR, "  1. mysql -u root -p -e \"CREATE DATABASE handsdb_clean\"\n");
    fwrite(STDERR, "  2. mysql -u root -p handsdb_clean < complete_export.sql\n\n");

} catch (PDOException $e) {
    fwrite(STDERR, "Database error: " . $e->getMessage() . "\n");
    exit(1);
}
