-- Migration to remove size, seller, and type references
-- Run this against your database

-- Helper procedure to drop foreign keys referencing a specific table
DROP PROCEDURE IF EXISTS `DropFKOnTableReferencing`;
DELIMITER $$
CREATE PROCEDURE `DropFKOnTableReferencing`(IN tableName VARCHAR(64), IN refTableName VARCHAR(64))
BEGIN
    DECLARE done INT DEFAULT FALSE;
    DECLARE fkName VARCHAR(64);
    DECLARE cur CURSOR FOR 
        SELECT constraint_name 
        FROM information_schema.key_column_usage
        WHERE table_name = tableName 
        AND referenced_table_name = refTableName 
        AND table_schema = DATABASE();
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = TRUE;

    OPEN cur;

    read_loop: LOOP
        FETCH cur INTO fkName;
        IF done THEN
            LEAVE read_loop;
        END IF;

        SET @s = CONCAT('ALTER TABLE `', tableName, '` DROP FOREIGN KEY `', fkName, '`');
        PREPARE stmt FROM @s;
        EXECUTE stmt;
        DEALLOCATE PREPARE stmt;
    END LOOP;

    CLOSE cur;
END$$
DELIMITER ;

-- 1. Drop Foreign Keys
-- stock_transfers -> product_sizes
CALL DropFKOnTableReferencing('stock_transfers', 'product_sizes');
-- inventory -> product_sizes
CALL DropFKOnTableReferencing('inventory', 'product_sizes');
-- stock_transactions -> product_sizes
CALL DropFKOnTableReferencing('stock_transactions', 'product_sizes');
-- stock_transactions -> sellers
CALL DropFKOnTableReferencing('stock_transactions', 'sellers');
-- products -> types
CALL DropFKOnTableReferencing('products', 'types');

-- 2. Drop Columns
-- stock_transfers.size_id
SET @exist := (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'stock_transfers' AND column_name = 'size_id' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE stock_transfers DROP COLUMN size_id', 'SELECT "Column size_id already dropped"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- inventory.size_id
-- First drop the unique index if it includes size_id (it likely does if size was part of uniqueness)
SET @exist := (SELECT COUNT(*) FROM information_schema.statistics WHERE table_name = 'inventory' AND index_name = 'unique_product_location' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE inventory DROP INDEX unique_product_location', 'SELECT "Index unique_product_location not found"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Drop the column
SET @exist := (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'inventory' AND column_name = 'size_id' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE inventory DROP COLUMN size_id', 'SELECT "Column size_id already dropped"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Recreate unique index on inventory (product_id, location_id)
-- Check if it exists first (it shouldn't, we just dropped it, but for safety)
SET @exist := (SELECT COUNT(*) FROM information_schema.statistics WHERE table_name = 'inventory' AND index_name = 'unique_product_location' AND table_schema = DATABASE());
SET @sql := IF(@exist = 0, 'ALTER TABLE inventory ADD UNIQUE KEY `unique_product_location` (`product_id`, `location_id`)', 'SELECT "Index unique_product_location already exists"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- stock_transactions.size_id
SET @exist := (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'stock_transactions' AND column_name = 'size_id' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE stock_transactions DROP COLUMN size_id', 'SELECT "Column size_id already dropped"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- stock_transactions.seller_id
SET @exist := (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'stock_transactions' AND column_name = 'seller_id' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE stock_transactions DROP COLUMN seller_id', 'SELECT "Column seller_id already dropped"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- products columns (product_type, type_id, available_sizes)
SET @exist := (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'products' AND column_name = 'product_type' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE products DROP COLUMN product_type', 'SELECT "Column product_type already dropped"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @exist := (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'products' AND column_name = 'type_id' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE products DROP COLUMN type_id', 'SELECT "Column type_id already dropped"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @exist := (SELECT COUNT(*) FROM information_schema.columns WHERE table_name = 'products' AND column_name = 'available_sizes' AND table_schema = DATABASE());
SET @sql := IF(@exist > 0, 'ALTER TABLE products DROP COLUMN available_sizes', 'SELECT "Column available_sizes already dropped"');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 3. Drop Tables
DROP TABLE IF EXISTS `product_sizes`;
DROP TABLE IF EXISTS `sellers`;
DROP TABLE IF EXISTS `types`;

-- Cleanup
DROP PROCEDURE IF EXISTS `DropFKOnTableReferencing`;

SELECT "Migration completed successfully";