# Database Export and Clean Import - Complete Guide

## Overview

This directory contains tools to export your current database and create a clean version without duplicate inventory records.

## Files Included

1. **export_database_with_fix.php** - Automated PHP export script
2. **COMPLETE_EXPORT_TEMPLATE.sql** - Manual SQL template
3. **EXPORT_INSTRUCTIONS.md** - Detailed step-by-step instructions
4. **create_clean_database.bat** - Windows automation script
5. **fix_duplicate_inventory.sql** - SQL-only duplicate fix

## Quick Start (Choose One Method)

### Method 1: Automated PHP Export (Easiest)

```bash
# 1. Update database credentials in export_database_with_fix.php if needed
# 2. Run the export
php export_database_with_fix.php > complete_export.sql

# 3. Create new database
mysql -u root -p -e "CREATE DATABASE handsdb_clean"

# 4. Import
mysql -u root -p handsdb_clean < complete_export.sql

# 5. Verify
mysql -u root -p handsdb_clean -e "SELECT COUNT(*) FROM inventory; SELECT product_id, location_id, COUNT(*) FROM inventory GROUP BY product_id, location_id HAVING COUNT(*) > 1"
```

### Method 2: Windows Batch Script (Fully Automated)

```cmd
# Just run this - it does everything
create_clean_database.bat
```

This will:
- ✓ Backup your current database
- ✓ Export with duplicate consolidation
- ✓ Create new clean database
- ✓ Import cleaned data
- ✓ Verify results

### Method 3: Manual mysqldump

```bash
# 1. Backup current database
mysqldump -u root -p handsdb > backup_$(date +%Y%m%d).sql

# 2. Export structure
mysqldump -u root -p --no-data handsdb > structure.sql

# 3. Export data (all tables except inventory)
mysqldump -u root -p --no-create-info handsdb \
  categories locations departments users products employees \
  toast_templates humor_lines stock_transfers stock_transactions \
  activity_logs notifications user_sessions > data.sql

# 4. Export consolidated inventory
mysql -u root -p handsdb -N -e "
SELECT 
    CONCAT('(',
    product_id, ',',
    location_id, ',',
    SUM(quantity), ',',
    MAX(min_quantity), ',',
    QUOTE(MAX(last_updated)), ',',
    COALESCE(QUOTE(GROUP_CONCAT(DISTINCT remarks SEPARATOR '; ')), 'NULL'),
    ')')
FROM inventory
GROUP BY product_id, location_id
" | sed 's/^/INSERT INTO inventory (product_id, location_id, quantity, min_quantity, last_updated, remarks) VALUES /' | sed 's/$/;/' > inventory_consolidated.sql

# 5. Create new database
mysql -u root -p -e "CREATE DATABASE handsdb_clean"

# 6. Import everything
mysql -u root -p handsdb_clean < structure.sql
mysql -u root -p handsdb_clean < data.sql
mysql -u root -p handsdb_clean < inventory_consolidated.sql
```

## What Gets Fixed

### Before (with duplicates):
```
| id | product_id | location_id | quantity |
|----|------------|-------------|----------|
| 1  | 5          | 2           | 50       | ← Was size S
| 2  | 5          | 2           | 50       | ← Was size M
```

### After (consolidated):
```
| id | product_id | location_id | quantity |
|----|------------|-------------|----------|
| 1  | 5          | 2           | 100      | ← Combined
```

## Switching to Clean Database

### Option A: Update Config
Edit `v3/config/config.php`:
```php
define('DB_NAME', 'handsdb_clean');
```

### Option B: Rename Databases
```bash
# Keep old as backup
mysql -u root -p -e "RENAME TABLE handsdb.* TO handsdb_old.*"

# Use clean as main
mysql -u root -p -e "RENAME TABLE handsdb_clean.* TO handsdb.*"
```

## Verification Steps

After import, verify:

```sql
-- 1. Check for duplicates (should return 0 rows)
SELECT product_id, location_id, COUNT(*) 
FROM inventory 
GROUP BY product_id, location_id 
HAVING COUNT(*) > 1;

-- 2. Verify unique constraint exists
SHOW INDEX FROM inventory WHERE Key_name = 'unique_product_location';

-- 3. Check total inventory
SELECT 
    COUNT(*) as total_records,
    SUM(quantity) as total_units
FROM inventory;

-- 4. Compare with old database
-- Run this on both databases and compare results
SELECT 
    p.part_number,
    l.name as location,
    SUM(i.quantity) as total_qty
FROM inventory i
JOIN products p ON i.product_id = p.id
JOIN locations l ON i.location_id = l.id
GROUP BY p.part_number, l.name
ORDER BY p.part_number, l.name;
```

## Troubleshooting

### "Duplicate entry" error
The unique constraint is working! This means you still have duplicates. Run the consolidation again.

### "Table doesn't exist"
Make sure you created the database first:
```bash
mysql -u root -p -e "CREATE DATABASE handsdb_clean"
```

### "Access denied"
Update credentials in the PHP script or use correct username:
```bash
mysql -u your_username -p
```

### "Can't connect to MySQL server"
Make sure MySQL is running:
```bash
# Windows
net start MySQL80

# Linux
sudo systemctl start mysql
```

## Backup Location

All backups are saved to: `Z:/hnsims/v3/database/backups/`

Format: `handsdb_backup_YYYYMMDD_HHMMSS.sql`

## Support Files

- **EXPORT_INSTRUCTIONS.md** - Detailed manual process
- **DUPLICATE_INVENTORY_FIX.md** - Explanation of the duplicate issue
- **fix_duplicate_inventory.sql** - SQL-only fix for existing database

## Need Help?

1. Check `EXPORT_INSTRUCTIONS.md` for detailed steps
2. Check `DUPLICATE_INVENTORY_FIX.md` for background on the issue
3. Review error messages carefully - they usually indicate the exact problem
4. Make sure you have a backup before making changes!
