# Duplicate Inventory Records - Root Cause and Fix

## Root Cause

The duplicate inventory records were caused by the migration that removed the `size_id` column from the inventory table (`migration_remove_size_seller_type.sql`).

### What Happened:

1. **Before Migration**: The inventory table had a unique constraint on `(product_id, location_id, size_id)`, allowing multiple records for the same product at the same location if they had different sizes.

2. **During Migration**: 
   - The `size_id` column was dropped
   - The unique constraint was dropped and recreated as `(product_id, location_id)`
   
3. **The Problem**: If there were already multiple inventory records for the same product-location with different sizes (e.g., "Example product 1" at "Bracebridge" in size S and size M), when the `size_id` column was dropped, those became duplicate records with identical `(product_id, location_id)` values.

4. **Constraint Failure**: When the migration tried to add back the unique constraint on `(product_id, location_id)`, it would **fail silently** if duplicates already existed, leaving the table without the constraint.

### Example:

**Before migration:**
```
| id | product_id | location_id | size_id | quantity |
|----|------------|-------------|---------|----------|
| 1  | 5          | 2           | 1       | 50       |
| 2  | 5          | 2           | 2       | 50       |
```

**After migration (with size_id removed):**
```
| id | product_id | location_id | quantity |
|----|------------|-------------|----------|
| 1  | 5          | 2           | 50       |  ← Duplicate!
| 2  | 5          | 2           | 50       |  ← Duplicate!
```

## Impact

This caused several issues:
1. Stock information panel showed "0 Units" because it only checked the first record
2. Stock out operations failed with "Insufficient stock" errors
3. Inventory reports showed incorrect quantities

## The Fix

We've implemented a two-part fix:

### Part 1: Code Fix (Already Applied)
Modified `InventoryModel.php` to handle duplicate records:
- `subtractStock()` now fetches ALL records for a product-location and sums quantities
- `getAvailableStock()` now uses `SUM(quantity)` to get total across duplicates
- Stock operations now work correctly even with duplicates

### Part 2: Database Cleanup (Run Once)

Choose one of the following methods:

#### Option A: Run PHP Script (Recommended)
```bash
cd Z:/hnsims/v3
php fix_inventory_duplicates.php
```

This script will:
1. Identify all duplicate records
2. Consolidate them by summing quantities
3. Ensure the unique constraint is in place
4. Show a summary of the changes

#### Option B: Run SQL Script
```bash
mysql -u [username] -p [database_name] < database/fix_duplicate_inventory.sql
```

## Prevention

After running the fix, the unique constraint `unique_product_location` on `(product_id, location_id)` will be in place, preventing future duplicates.

The `updateStock()` method uses `INSERT ... ON DUPLICATE KEY UPDATE`, which will properly handle the constraint and update existing records instead of creating duplicates.

## Verification

After running the fix, verify:

1. **No duplicates exist:**
```sql
SELECT product_id, location_id, COUNT(*) as count
FROM inventory
GROUP BY product_id, location_id
HAVING COUNT(*) > 1;
```
Should return 0 rows.

2. **Unique constraint exists:**
```sql
SHOW INDEX FROM inventory WHERE Key_name = 'unique_product_location';
```
Should show the constraint.

3. **Stock operations work:**
- Try viewing stock levels - should show correct totals
- Try stock out operation - should work without "insufficient stock" errors
