<?php

namespace App\Models;
use PDO;
use Throwable;

use App\Core\Model;

class InventoryModel extends Model
{
    protected $table = 'inventory';

    public function getStockByLocation($locationId = null)
    {
        // Group by product_id and location_id to consolidate duplicates
        // Sum quantities to show actual current stock levels
        $sql = "SELECT 
                i.product_id, 
                i.location_id, 
                p.part_number, 
                p.name as product_name, 
                p.description as product_desc, 
                p.low_stock_threshold, 
                p.deleted_at,
                l.name as location_name, 
                c.name as category_name,
                SUM(i.quantity) as quantity,
                MAX(i.min_quantity) as min_quantity,
                MAX(i.last_updated) as last_updated
                FROM inventory i
                JOIN products p ON i.product_id = p.id
                JOIN locations l ON i.location_id = l.id
                LEFT JOIN categories c ON p.category_id = c.id";

        if ($locationId) {
            $sql .= " WHERE i.location_id = :location_id";
        }

        $sql .= " GROUP BY i.product_id, i.location_id, p.part_number, p.name, p.description, p.low_stock_threshold, p.deleted_at, l.name, c.name";
        $sql .= " ORDER BY l.name, p.part_number";

        $stmt = $this->db->prepare($sql);

        if ($locationId) {
            $stmt->execute(['location_id' => $locationId]);
        } else {
            $stmt->execute();
        }

        return $stmt->fetchAll();
    }

    public function updateStock($productId, $locationId, $quantity, $type = 'add')
    {
        // Use atomic database operations to prevent race conditions.

        // Sanitize quantity to ensure it's a positive integer.
        $quantity = abs((int)$quantity);

        // For subtractions, use the new subtractStock method
        if ($type !== 'add') {
            $result = $this->subtractStock($productId, $locationId, $quantity);
            return $result['ok'];
        }

        // For additions, use INSERT ... ON DUPLICATE KEY UPDATE for atomicity.
        // This requires a UNIQUE constraint on (product_id, location_id).
        // Assumes such a constraint exists.
        $sql = "INSERT INTO inventory (product_id, location_id, quantity) 
                VALUES (:product_id, :location_id, :quantity)
                ON DUPLICATE KEY UPDATE quantity = quantity + :quantity";

        $stmt = $this->db->prepare($sql);

        return $stmt->execute(
            [
            'product_id' => $productId,
            'location_id' => $locationId,
            'quantity' => $quantity
            ]
        );
    }

    /**
     * Subtract stock with proper validation and locking
     * Fails if insufficient stock available
     * Handles duplicate inventory records by summing quantities
     */
    public function subtractStock(int $productId, int $locationId, int $quantity): array
    {
        $db = $this->db;

        // Check if a transaction is already active
        $transactionStartedHere = false;
        if (!$db->inTransaction()) {
            $db->beginTransaction();
            $transactionStartedHere = true;
        }

        try {
            // Lock all rows for this product-location to prevent race conditions
            $sql = "SELECT id, quantity FROM inventory
                    WHERE product_id = :p AND location_id = :l
                    FOR UPDATE";

            $stmt = $db->prepare($sql);
            $params = ['p' => $productId, 'l' => $locationId];
            $stmt->execute($params);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Calculate total available stock across all duplicate records
            $totalAvailable = 0;
            foreach ($rows as $row) {
                $totalAvailable += (int)$row['quantity'];
            }

            // Check if we have sufficient stock
            if (empty($rows) || $totalAvailable < $quantity) {
                if ($transactionStartedHere) {
                    $db->rollBack();
                }
                return [
                    'ok' => false,
                    'error' => 'INSUFFICIENT_STOCK',
                    'available' => $totalAvailable,
                    'requested' => $quantity
                ];
            }

            // Subtract from records in order until quantity is fulfilled
            $remainingToSubtract = $quantity;
            $upd = $db->prepare("UPDATE inventory SET quantity = :new_qty WHERE id = :id");
            
            foreach ($rows as $row) {
                $rowQty = (int)$row['quantity'];
                
                if ($remainingToSubtract <= 0) {
                    break;
                }
                
                if ($rowQty >= $remainingToSubtract) {
                    // This row has enough to fulfill the remaining quantity
                    $newQty = $rowQty - $remainingToSubtract;
                    $upd->execute(['new_qty' => $newQty, 'id' => $row['id']]);
                    $remainingToSubtract = 0;
                } else {
                    // This row doesn't have enough, use all of it
                    $upd->execute(['new_qty' => 0, 'id' => $row['id']]);
                    $remainingToSubtract -= $rowQty;
                }
            }

            if ($transactionStartedHere) {
                $db->commit();
            }
            return ['ok' => true];
        } catch (Throwable $e) {
            if ($transactionStartedHere) {
                $db->rollBack();
            }
            throw $e;
        }
    }

    public function getLowStockItems(?int $locationId = null, ?string $dateFrom = null, ?string $dateTo = null)
    {
        return $this->getLowStockItemsDetailed($locationId, $dateFrom, $dateTo);
    }

    public function getLowStockItemsDetailed(?int $locationId = null, ?string $dateFrom = null, ?string $dateTo = null)
    {
        // Use subquery to group duplicates first, then filter for low stock
        $sql = "SELECT 
                inv.product_id, 
                inv.location_id, 
                p.part_number, 
                p.name as product_name, 
                p.description as product_desc, 
                p.low_stock_threshold, 
                p.deleted_at, 
                l.name as location_name, 
                c.name as category_name, 
                inv.quantity, 
                inv.min_quantity, 
                inv.last_updated
                FROM (
                    SELECT 
                        product_id,
                        location_id,
                        SUM(quantity) as quantity,
                        MAX(min_quantity) as min_quantity,
                        MAX(last_updated) as last_updated
                    FROM inventory
                    GROUP BY product_id, location_id
                ) inv
                JOIN products p ON inv.product_id = p.id
                JOIN locations l ON inv.location_id = l.id
                LEFT JOIN categories c ON p.category_id = c.id
                WHERE inv.quantity <= COALESCE(p.low_stock_threshold, inv.min_quantity)";

        $params = [];
        if ($locationId) {
            $sql .= " AND inv.location_id = :location_id";
            $params['location_id'] = $locationId;
        }

        if (!empty($dateFrom)) {
            $sql .= " AND DATE(inv.last_updated) >= :date_from";
            $params['date_from'] = $dateFrom;
        }

        if (!empty($dateTo)) {
            $sql .= " AND DATE(inv.last_updated) <= :date_to";
            $params['date_to'] = $dateTo;
        }

        $sql .= " ORDER BY inv.quantity ASC";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public function getTotalStockQuantity()
    {
        $sql = "SELECT SUM(quantity) as total_quantity FROM inventory";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)($result['total_quantity'] ?? 0);
    }

    public function getAvailableStock(int $productId, int $locationId): int
    {
        $sql = "SELECT SUM(quantity) as total_quantity FROM inventory 
                WHERE product_id = :product_id AND location_id = :location_id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['product_id' => $productId, 'location_id' => $locationId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? (int)($result['total_quantity'] ?? 0) : 0;
    }
}
