<?php

namespace App\Helpers;

use App\Models\NotificationModel;

/**
 * Notification Helper
 * Creates smart, contextual notifications for important events
 */

class NotificationHelper
{
    private $db;
    private $notificationModel;

    public function __construct($db)
    {
        $this->db = $db;
        $this->notificationModel = new NotificationModel($db);
    }

    /**
     * Check inventory levels and create notifications for low/critical stock
     */
    public function checkInventoryLevels()
    {
        $sql = "SELECT 
                    i.id,
                    i.product_id,
                    i.location_id,
                    i.quantity,
                    p.part_number,
                    p.name as product_name,
                    p.low_stock_threshold,
                    l.name as location_name,
                    CASE 
                        WHEN i.quantity = 0 THEN 'out_of_stock'
                        WHEN i.quantity <= (p.low_stock_threshold * 0.5) THEN 'critical'
                        WHEN i.quantity <= p.low_stock_threshold THEN 'low'
                        ELSE 'normal'
                    END as stock_status
                FROM inventory i
                JOIN products p ON i.product_id = p.id
                JOIN locations l ON i.location_id = l.id
                WHERE i.quantity <= p.low_stock_threshold
                AND p.low_stock_threshold > 0
                ORDER BY stock_status DESC, i.quantity ASC";

        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        $items = $stmt->fetchAll();

        foreach ($items as $item) {
            $this->createInventoryAlert($item);
        }
    }

    /**
     * Create inventory alert notification
     */
    private function createInventoryAlert($item)
    {
        $productName = $item['product_name'] ?: $item['part_number'];
        $location = $item['location_name'];
        $quantity = $item['quantity'];
        $minQty = $item['low_stock_threshold'];
        $message = '';
        $type = 'inventory';

        switch ($item['stock_status']) {
            case 'out_of_stock':
                $message = "⚠️ OUT OF STOCK: {$productName} at {$location}. Restock immediately!";
                break;
            case 'critical':
                $message = "🔴 CRITICAL: Only {$quantity} units of {$productName} left at {$location} (Min: {$minQty})";
                break;
            case 'low':
                $message = "🟡 LOW STOCK: {$productName} at {$location} has {$quantity} units (Min: {$minQty})";
                break;
        }

        if ($message) {
            // Notify all admin and manager users
            $usersSql = "SELECT id FROM users WHERE is_active = 1 AND role IN ('admin', 'manager')";
            $usersStmt = $this->db->prepare($usersSql);
            $usersStmt->execute();
            $users = $usersStmt->fetchAll();

            foreach ($users as $user) {
                $this->notificationModel->addNotification(
                    $user['id'],
                    $message,
                    $type
                );
            }
        }
    }

    /**
     * Create notification for stock added
     */
    public function notifyStockAdded($userId, $username, $productName, $quantity, $location)
    {
        $message = "✅ {$username} added {$quantity} units of {$productName} to {$location}";

        // Only notify other users, not the one who performed the action
        $this->notifyOtherUsers($userId, $message, 'activity');
    }

    /**
     * Create notification for stock removed
     */
    public function notifyStockRemoved($userId, $username, $productName, $quantity, $location, $department)
    {
        $deptText = $department ? " for {$department}" : '';
        $message = "📤 {$username} removed {$quantity} units of {$productName} from {$location}{$deptText}";

        $this->notifyOtherUsers($userId, $message, 'activity');
    }

    /**
     * Create notification for stock transfer
     */
    public function notifyStockTransferred($userId, $username, $productName, $quantity, $fromLocation, $toLocation)
    {
        $message = "🔄 {$username} transferred {$quantity} units of {$productName} from {$fromLocation} to {$toLocation}";

        $this->notifyOtherUsers($userId, $message, 'activity');
    }

    /**
     * Create notification for new product added
     */
    public function notifyProductAdded($userId, $username, $productName, $partNumber)
    {
        $message = "🆕 {$username} added new product: {$productName} ({$partNumber})";

        $this->notifyOtherUsers($userId, $message, 'system');
    }

    /**
     * Create notification for product deleted
     */
    public function notifyProductDeleted($userId, $username, $productName)
    {
        $message = "🗑️ {$username} deleted product: {$productName}";

        $this->notifyOtherUsers($userId, $message, 'system');
    }

    /**
     * Notify all users except the one who performed the action
     */
    private function notifyOtherUsers($excludeUserId, $message, $type)
    {
        $sql = "SELECT id FROM users WHERE is_active = 1 AND id != :exclude_id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['exclude_id' => $excludeUserId]);
        $users = $stmt->fetchAll();

        foreach ($users as $user) {
            $this->notificationModel->addNotification($user['id'], $message, $type);
        }
    }

    /**
     * Notify specific user
     */
    public function notifyUser($userId, $message, $type = 'system')
    {
        $this->notificationModel->addNotification($userId, $message, $type);
    }

    /**
     * Notify all users
     */
    public function notifyAllUsers($message, $type = 'system')
    {
        $sql = "SELECT id FROM users WHERE is_active = 1";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        $users = $stmt->fetchAll();

        foreach ($users as $user) {
            $this->notificationModel->addNotification($user['id'], $message, $type);
        }
    }
}
