<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Helpers\NotificationHelper;
use InvalidArgumentException;
use RuntimeException;

require_once __DIR__ . '/../Helpers/csrf.php';

class StockController extends Controller
{
    public function add()
    {
        $this->requireAuth();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // CSRF Protection
            csrf_check();

            try {
                $productId = (int)($_POST['product_id'] ?? 0);
                $locationId = (int)($_POST['location_id'] ?? 0);
                $quantity = (int)($_POST['quantity'] ?? 0);
                $remarks = $_POST['remarks'] ?? '';
                $pricePerUnit = isset($_POST['price_per_unit']) && $_POST['price_per_unit'] !== '' ? $_POST['price_per_unit'] : null;

                $data = [
                    'product_id' => $productId,
                    'location_id' => $locationId,
                    'quantity' => $quantity,
                    'remarks' => $remarks,
                    'price_per_unit' => $pricePerUnit
                ];

                $errors = $this->validateStockData($data, 'add', $this->model('ProductModel'));
                if (!empty($errors)) {
                    throw new InvalidArgumentException(implode(' ', $errors));
                }

                // Update inventory
                $inventoryModel = $this->model('InventoryModel');
                $inventoryModel->updateStock($productId, $locationId, $quantity, 'add');

                // Add transaction record
                $transactionModel = $this->model('TransactionModel');
                $transactionModel->addTransaction(
                    [
                    'transaction_type' => 'IN',
                    'product_id' => $productId,
                    'location_id' => $locationId,
                    'department_id' => null,
                    'employee_id' => null,
                    'assignment_type' => 'department',
                    'quantity' => $quantity,
                    'user_id' => $_SESSION['user_id'],
                    'remarks' => $remarks,
                    'price_per_unit' => $pricePerUnit
                    ]
                );

                // Add log
                $productModel = $this->model('ProductModel');
                $product = $productModel->getById($productId);
                $locationName = $this->getLocationName($locationId);

                $logDetails = $_SESSION['username'] . " added " . $quantity . " " .
                             $product['name'] . " to " . $locationName . " warehouse";
                $this->addLog("Stock added", $logDetails);

                // Create notification for other users
                $notificationHelper = new NotificationHelper($this->db);
                $notificationHelper->notifyStockAdded(
                    $_SESSION['user_id'],
                    $_SESSION['username'],
                    $product['name'],
                    '',
                    $quantity,
                    $locationName
                );

                // Check inventory levels after adding stock
                $notificationHelper->checkInventoryLevels();

                // Personalized success message using toast template
                $toastTemplateModel = $this->model('ToastTemplateModel');
                $_SESSION['success'] = $toastTemplateModel->formatMessage(
                    'stock_add_success', [
                    '%q' => $quantity,
                    '%s' => '',
                    '%p' => $product['name'],
                    '%l' => $locationName
                    ]
                );
            } catch (\Exception $e) {
                $_SESSION['error'] = 'Failed to add stock: ' . $e->getMessage();
            }

            $this->redirect('/stock/add');
        }

        // Get data for form
        $data = [
            'products' => $this->model('ProductModel')->getAll(),
            'locations' => $this->getLocations(),
            'categories' => $this->getCategories()
        ];

        // Pre-load data from query parameters
        $selectedProduct = null;
        $selectedLocation = null;

        if (isset($_GET['product_id'])) {
            $productModel = $this->model('ProductModel');
            $selectedProduct = $productModel->getById((int)$_GET['product_id']);
        }

        if (isset($_GET['location_id'])) {
            $locationId = (int)$_GET['location_id'];
            $selectedLocation = array_filter(
                $data['locations'], function ($loc) use ($locationId) {
                    return $loc['id'] == $locationId;
                }
            );
            $selectedLocation = !empty($selectedLocation) ? reset($selectedLocation) : null;
        }

        $data['selected_product'] = $selectedProduct;
        $data['selected_location'] = $selectedLocation;

        $this->view('stock/add', $data);
    }

    public function out()
    {
        $this->requireAuth();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // CSRF Protection
            csrf_check();
            $productId = (int)($_POST['product_id'] ?? 0);
            $locationId = $_POST['location_id'] ?? '';
            $assignmentType = $_POST['assignment_type'] ?? 'department';
            $departmentId = !empty($_POST['department_id']) ? (int)$_POST['department_id'] : null;
            $employeeId = !empty($_POST['employee_id']) ? (int)$_POST['employee_id'] : null;
            $quantity = $_POST['quantity'] ?? 0;
            $remarks = $_POST['remarks'] ?? '';

            $data = [
                'product_id' => $productId,
                'location_id' => $locationId,
                'department_id' => $departmentId,
                'employee_id' => $employeeId,
                'assignment_type' => $assignmentType,
                'quantity' => $quantity,
                'remarks' => $remarks
            ];

            $errors = $this->validateStockData($data, 'out', $this->model('ProductModel'));
            if (!empty($errors)) {
                $_SESSION['error'] = implode(' ', $errors);
                $this->redirect('/stock/out');
                return;
            }

            // Update inventory with proper error handling
            $inventoryModel = $this->model('InventoryModel');
            $result = $inventoryModel->subtractStock($productId, $locationId, $quantity);

            if (!$result['ok']) {
                $errorMsg = 'Insufficient stock at selected location. ';
                if (isset($result['available'])) {
                    $errorMsg .= 'Available: ' . $result['available'] . ', Requested: ' . $result['requested'];
                }
                $_SESSION['error'] = $errorMsg;
                $this->redirect('/stock/out');
                return;
            }

            // Add transaction record with employee tracking
            $transactionModel = $this->model('TransactionModel');
            $transactionModel->addTransaction(
                [
                'transaction_type' => 'OUT',
                'product_id' => $productId,
                'location_id' => $locationId,
                'department_id' => $departmentId,
                'employee_id' => $employeeId,
                'assignment_type' => $assignmentType,
                'quantity' => $quantity,
                'user_id' => $_SESSION['user_id'],
                'remarks' => $remarks,
                'price_per_unit' => null
                ]
            );

            // Add log
            $productModel = $this->model('ProductModel');
            $product = $productModel->getById($productId);
            $locationName = $this->getLocationName($locationId);
            
            // Get recipient name based on assignment type
            $recipientName = '';
            if ($assignmentType === 'employee' && $employeeId) {
                $recipientName = $this->getEmployeeName($employeeId);
                $logDetails = $_SESSION['username'] . " gave " . $quantity . " " .
                             $product['name'] . " to employee " . $recipientName . " from " . $locationName;
            } else {
                $recipientName = $this->getDepartmentName($departmentId);
                $logDetails = $_SESSION['username'] . " gave " . $quantity . " " .
                             $product['name'] . " to department " . $recipientName . " from " . $locationName;
            }
            
            $this->addLog("Stock removed", $logDetails);

            // Create notification for other users
            $notificationHelper = new NotificationHelper($this->db);
            $notificationHelper->notifyStockRemoved(
                $_SESSION['user_id'],
                $_SESSION['username'],
                $product['name'],
                '',
                $quantity,
                $locationName,
                $recipientName
            );

            // Check inventory levels after removing stock (IMPORTANT!)
            $notificationHelper->checkInventoryLevels();

            // Personalized success message using toast template
            $toastTemplateModel = $this->model('ToastTemplateModel');
            $_SESSION['success'] = $toastTemplateModel->formatMessage(
                'stock_out_success', [
                '%q' => $quantity,
                '%s' => '',
                '%p' => $product['name'],
                '%d' => $recipientName,
                '%l' => $locationName
                ]
            );
            $this->redirect('/stock/out');
        }

        // Get data for form
        $employeeModel = $this->model('EmployeeModel');
        $data = [
            'products' => $this->model('ProductModel')->getAll(),
            'locations' => $this->getLocations(),
            'departments' => $this->getDepartments(),
            'employees' => $employeeModel->getAllEmployees(),
            'categories' => $this->getCategories()
        ];

        // Pre-load data from query parameters
        $selectedProduct = null;
        $selectedLocation = null;

        if (isset($_GET['product_id'])) {
            $productModel = $this->model('ProductModel');
            $selectedProduct = $productModel->getById((int)$_GET['product_id']);
        }

        if (isset($_GET['location_id'])) {
            $locationId = (int)$_GET['location_id'];
            $selectedLocation = array_filter(
                $data['locations'], function ($loc) use ($locationId) {
                    return $loc['id'] == $locationId;
                }
            );
            $selectedLocation = !empty($selectedLocation) ? reset($selectedLocation) : null;
        }

        $data['selected_product'] = $selectedProduct;
        $data['selected_location'] = $selectedLocation;

        $this->view('stock/out', $data);
    }

    public function inStock()
    {
        $this->requireAuth();

        $inventoryModel = $this->model('InventoryModel');
        $locations = $this->getLocations();

        $selectedLocation = $_GET['location'] ?? null;
        $filter = $_GET['filter'] ?? null;

        if ($filter === 'low_stock') {
            $stockData = $inventoryModel->getLowStockItemsDetailed($selectedLocation);
        } else {
            $stockData = $inventoryModel->getStockByLocation($selectedLocation);
        }

        $data = [
            'locations' => $locations,
            'stock' => $stockData,
            'selected_location' => $selectedLocation,
            'filter' => $filter
        ];

        // Check if it's an AJAX request
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            $this->partial('stock/partials/stock_content', $data);
        } else {
            $this->view('stock/in-stock', $data);
        }
    }

    public function transfer()
    {
        $this->requireAuth();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // CSRF Protection
            csrf_check();
            $productId = (int)($_POST['product_id'] ?? 0);
            $fromLocationId = $_POST['from_location_id'] ?? '';
            $toLocationId = $_POST['to_location_id'] ?? '';
            $quantity = (int)($_POST['quantity'] ?? 0);
            $remarks = $_POST['remarks'] ?? '';

            $data = [
                'product_id' => $productId,
                'location_id' => $fromLocationId,
                'quantity' => $quantity
            ];

            $errors = $this->validateStockData($data, 'transfer', $this->model('ProductModel'));
            if ($toLocationId === '' || $toLocationId <= 0) {
                $errors[] = 'Please select a valid destination location.';
            }
            if ($fromLocationId == $toLocationId) {
                $errors[] = 'Source and destination locations must be different.';
            }

            if (!empty($errors)) {
                $_SESSION['error'] = implode(' ', $errors);
                return $this->redirect('/stock/transfer');
            }

            // Atomic transfer with proper transaction handling
            $inventoryModel = $this->model('InventoryModel');
            $transactionModel = $this->model('TransactionModel');
            $productModel = $this->model('ProductModel');

            $db = $this->db;
            $db->beginTransaction();

            try {
                // First check if we have sufficient stock at source
                $result = $inventoryModel->subtractStock($productId, $fromLocationId, $quantity);

                if (!$result['ok']) {
                    throw new RuntimeException(
                        'Insufficient stock at source location. Available: ' .
                        ($result['available'] ?? 0) . ', Requested: ' . $quantity
                    );
                }

                // Add to destination
                $inventoryModel->updateStock($productId, $toLocationId, $quantity, 'add');

                // Get names for logging
                $product = $productModel->getById($productId);
                $fromLocationName = $this->getLocationName($fromLocationId);
                $toLocationName = $this->getLocationName($toLocationId);

                $transferRemark = "transfer\n" . $fromLocationName . " -> " . $toLocationName;
                if (!empty($remarks)) {
                    $transferRemark .= "\n" . $remarks;
                }

                // Record transaction as two entries for traceability
                $transactionModel->addTransaction(
                    [
                    'transaction_type' => 'OUT',
                    'product_id' => $productId,
                    'location_id' => $fromLocationId,
                    'department_id' => null,
                    'employee_id' => null,
                    'assignment_type' => 'department',
                    'quantity' => $quantity,
                    'user_id' => $_SESSION['user_id'],
                    'remarks' => $transferRemark,
                    'price_per_unit' => null
                    ]
                );

                $transactionModel->addTransaction(
                    [
                    'transaction_type' => 'IN',
                    'product_id' => $productId,
                    'location_id' => $toLocationId,
                    'department_id' => null,
                    'employee_id' => null,
                    'assignment_type' => 'department',
                    'quantity' => $quantity,
                    'user_id' => $_SESSION['user_id'],
                    'remarks' => $transferRemark,
                    'price_per_unit' => null
                    ]
                );

                $db->commit();

                // Log the transfer
                $logDetails = $_SESSION['username'] . " transferred " . $quantity . " " .
                             $product['name'] . " from " . $fromLocationName . " to " . $toLocationName;
                $this->addLog("Stock transferred", $logDetails);

                // Create notification for other users
                $notificationHelper = new NotificationHelper($this->db);
                $notificationHelper->notifyStockTransferred(
                    $_SESSION['user_id'],
                    $_SESSION['username'],
                    $product['name'],
                    '',
                    $quantity,
                    $fromLocationName,
                    $toLocationName
                );

                // Check inventory levels after transfer
                $notificationHelper->checkInventoryLevels();

                // Personalized success message using toast template
                $toastTemplateModel = $this->model('ToastTemplateModel');
                $_SESSION['success'] = $toastTemplateModel->formatMessage(
                    'stock_transfer_success', [
                    '%q' => $quantity,
                    '%s' => '',
                    '%p' => $product['name'],
                    '%f' => $fromLocationName,
                    '%t' => $toLocationName
                    ]
                );
            } catch (\Exception $e) {
                if ($db->inTransaction()) {
                    $db->rollBack();
                }
                $_SESSION['error'] = 'Transfer failed: ' . $e->getMessage();
                return $this->redirect('/stock/transfer');
            }
        }

        $data = [
            'products' => $this->model('ProductModel')->getAll(),
            'locations' => $this->getLocations(),
            'categories' => $this->getCategories()
        ];

        // Pre-load data from query parameters
        $selectedProduct = null;
        $selectedFromLocation = null;

        if (isset($_GET['product_id'])) {
            $productModel = $this->model('ProductModel');
            $selectedProduct = $productModel->getById((int)$_GET['product_id']);
        }

        if (isset($_GET['from_location_id'])) {
            $locationId = (int)$_GET['from_location_id'];
            $selectedFromLocation = array_filter(
                $data['locations'], function ($loc) use ($locationId) {
                    return $loc['id'] == $locationId;
                }
            );
            $selectedFromLocation = !empty($selectedFromLocation) ? reset($selectedFromLocation) : null;
        }

        $data['selected_product'] = $selectedProduct;
        $data['selected_from_location'] = $selectedFromLocation;

        $this->view('stock/transfer', $data);
    }

    private function getLocations()
    {
        $sql = "SELECT * FROM locations ORDER BY name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    private function getDepartments()
    {
        $sql = "SELECT * FROM departments ORDER BY name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    private function getCategories()
    {
        $sql = "SELECT * FROM categories ORDER BY name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    private function getLocationName($id)
    {
        $sql = "SELECT name FROM locations WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['id' => $id]);
        $result = $stmt->fetch();
        return $result ? $result['name'] : 'Unknown';
    }

    private function getDepartmentName($id)
    {
        $sql = "SELECT name FROM departments WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['id' => $id]);
        $result = $stmt->fetch();
        return $result ? $result['name'] : 'Unknown';
    }

    private function getEmployeeName($id)
    {
        $sql = "SELECT name FROM employees WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['id' => $id]);
        $result = $stmt->fetch();
        return $result ? $result['name'] : 'Unknown';
    }

    public function getInventoryJson()
    {
        $this->requireAuth();
        $inventoryModel = $this->model('InventoryModel');
        $stockData = $inventoryModel->getStockByLocation();

        header('Content-Type: application/json');
        echo json_encode($stockData);
        exit;
    }

    public function bulkAdd()
    {
        $this->requireAuth();
        csrf_check();

        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('/stock/add');
            return;
        }

        if (!isset($_FILES['bulk_file']) || $_FILES['bulk_file']['error'] !== UPLOAD_ERR_OK) {
            $_SESSION['error'] = 'File upload error. Please try again.';
            $this->redirect('/stock/add');
            return;
        }

        $maxFileSize = 10 * 1024 * 1024; // 10MB
        if ($_FILES['bulk_file']['size'] > $maxFileSize) {
            $_SESSION['error'] = 'File is too large. Maximum size is 10MB.';
            $this->redirect('/stock/add');
            return;
        }

        $file = $_FILES['bulk_file']['tmp_name'];
        $fileName = $_FILES['bulk_file']['name'];
        $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        
        $productModel = $this->model('ProductModel');
        $inventoryModel = $this->model('InventoryModel');
        $transactionModel = $this->model('TransactionModel');

        // Pre-load master data for faster lookup
        $allLocations = $this->getLocations();
        $locationMap = [];
        foreach ($allLocations as $location) {
            $locationMap[strtoupper($location['name'])] = $location['id'];
        }

        $rowCount = 0;
        $successCount = 0;
        $errorCount = 0;
        $errors = [];

        $this->db->beginTransaction();

        try {
            // Handle Excel files (.xlsx)
            if ($fileExtension === 'xlsx') {
                require_once __DIR__ . '/../../vendor/autoload.php';
                
                $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file);
                $worksheet = $spreadsheet->getSheet(0); // Get first sheet (Stock)
                $rows = $worksheet->toArray();
                
                // Skip header row
                array_shift($rows);
                
                foreach ($rows as $row) {
                    $rowCount++;
                    
                    // Skip empty rows
                    if (empty($row[0])) {
                        continue;
                    }
                    
                    // Map columns: Product Name, Location, Quantity, Remarks, Price Per Unit
                    $productName = trim($row[0] ?? '');
                    $locationName = trim($row[1] ?? '');
                    $quantity = !empty($row[2]) ? (int)$row[2] : 0;
                    $remarks = trim($row[3] ?? '');
                    $pricePerUnit = isset($row[4]) && $row[4] !== '' ? $row[4] : null;
                    
                    // Basic validation
                    if (empty($productName) || empty($locationName) || $quantity <= 0) {
                        $errorCount++;
                        $errors[] = "Row {$rowCount}: Product name, location, and quantity are required.";
                        continue;
                    }
                    
                    // Lookup product by name
                    $product = $productModel->getByName($productName);
                    if (!$product) {
                        $errorCount++;
                        $errors[] = "Row {$rowCount}: Product '{$productName}' not found.";
                        continue;
                    }
                    $productId = $product['id'];
                    
                    // Lookup location
                    $locationKey = strtoupper($locationName);
                    if (!isset($locationMap[$locationKey])) {
                        $errorCount++;
                        $errors[] = "Row {$rowCount}: Location '{$locationName}' not found.";
                        continue;
                    }
                    $locationId = $locationMap[$locationKey];
                    
                    // Update inventory
                    $inventoryModel->updateStock($productId, $locationId, $quantity, 'add');
                    
                    // Add transaction record
                    $transactionModel->addTransaction([
                        'transaction_type' => 'IN',
                        'product_id' => $productId,
                        'location_id' => $locationId,
                        'department_id' => null,
                        'employee_id' => null,
                        'assignment_type' => 'department',
                        'quantity' => $quantity,
                        'user_id' => $_SESSION['user_id'],
                        'remarks' => "Bulk add: " . $remarks,
                        'price_per_unit' => $pricePerUnit
                    ]);
                    
                    $successCount++;
                }
            } else {
                throw new \Exception('Invalid file type. Please upload an Excel (.xlsx) file.');
            }

            if ($errorCount > 0) {
                throw new \Exception("Errors occurred during bulk import.");
            }

            $this->db->commit();
            $_SESSION['success'] = "Bulk stock import finished. {$successCount} records imported successfully.";
        } catch (\Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            $message = "Bulk import failed. " . $e->getMessage();
            if ($errorCount > 0) {
                $_SESSION['error'] = $message . " {$errorCount} rows failed. Errors: " . implode('; ', $errors);
            } else {
                $_SESSION['error'] = $message;
            }
        }

        $this->redirect('/stock/in-stock');
    }

    public function checkStock()
    {
        $this->requireAuth();

        header('Content-Type: application/json');

        $productId = (int)($_GET['product_id'] ?? 0);
        $locationId = (int)($_GET['location_id'] ?? 0);

        if ($productId <= 0 || $locationId <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid product or location ID.']);
            exit;
        }

        $inventoryModel = $this->model('InventoryModel');
        $available = $inventoryModel->getAvailableStock($productId, $locationId);

        echo json_encode(['success' => true, 'available' => $available]);
        exit;
    }

    private function validateStockData($data, $type = 'add', $productModel = null)
    {
        $errors = [];

        if (empty($data['product_id']) || !is_numeric($data['product_id']) || (int)$data['product_id'] <= 0) {
            $errors[] = "A valid product must be selected.";
        }

        if (empty($data['location_id']) || !is_numeric($data['location_id']) || (int)$data['location_id'] <= 0) {
            $errors[] = "A valid location must be selected.";
        }

        if (empty($data['quantity']) || !is_numeric($data['quantity']) || (int)$data['quantity'] <= 0) {
            $errors[] = "Quantity must be a positive number.";
        }

        if ($type === 'out') {
            $assignmentType = $data['assignment_type'] ?? 'department';
            
            if ($assignmentType === 'employee') {
                if (empty($data['employee_id']) || !is_numeric($data['employee_id']) || (int)$data['employee_id'] <= 0) {
                    $errors[] = "A valid employee must be selected.";
                }
            } else {
                if (empty($data['department_id']) || !is_numeric($data['department_id']) || (int)$data['department_id'] <= 0) {
                    $errors[] = "A valid department must be selected.";
                }
            }
        }

        if ($type === 'add') {
            if (isset($data['price_per_unit']) && !empty($data['price_per_unit']) && !is_numeric($data['price_per_unit'])) {
                $errors[] = "Price per unit must be a valid number.";
            }
        }

        if (isset($data['remarks']) && strlen($data['remarks']) > 1000) {
            $errors[] = "Remarks cannot exceed 1000 characters.";
        }

        return $errors;
    }
}
