# Database Export and Clean Import Instructions

## Quick Start (Automated)

### Option 1: Using PHP Export Script (Recommended)
```bash
# 1. Export your current database with duplicate consolidation
php export_database_with_fix.php > complete_export.sql

# 2. Create new clean database
mysql -u root -p -e "CREATE DATABASE handsdb_clean CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci"

# 3. Import the cleaned data
mysql -u root -p handsdb_clean < complete_export.sql

# 4. Verify no duplicates
mysql -u root -p handsdb_clean -e "SELECT product_id, location_id, COUNT(*) FROM inventory GROUP BY product_id, location_id HAVING COUNT(*) > 1"
```

### Option 2: Using Batch Script (Windows)
```cmd
create_clean_database.bat
```
This will automatically:
- Backup your current database
- Export with duplicate consolidation
- Create new clean database
- Import cleaned data
- Verify the results

## Manual Process

### Step 1: Backup Current Database
```bash
mysqldump -u root -p handsdb > backup_handsdb_$(date +%Y%m%d).sql
```

### Step 2: Export Current Data
```bash
# Export structure
mysqldump -u root -p --no-data handsdb > structure.sql

# Export data from all tables EXCEPT inventory
mysqldump -u root -p --no-create-info handsdb \
  categories locations departments users products employees \
  toast_templates humor_lines stock_transfers stock_transactions \
  activity_logs notifications user_sessions > data_without_inventory.sql

# Export inventory with duplicate consolidation
mysql -u root -p handsdb -e "
SELECT 
    product_id,
    location_id,
    SUM(quantity) as quantity,
    MAX(min_quantity) as min_quantity,
    MAX(last_updated) as last_updated,
    GROUP_CONCAT(DISTINCT remarks SEPARATOR '; ') as remarks
FROM inventory
GROUP BY product_id, location_id
INTO OUTFILE '/tmp/inventory_consolidated.csv'
FIELDS TERMINATED BY ',' 
ENCLOSED BY '\"'
LINES TERMINATED BY '\n';
"
```

### Step 3: Create New Database
```bash
mysql -u root -p -e "CREATE DATABASE handsdb_clean CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci"
```

### Step 4: Import Structure
```bash
mysql -u root -p handsdb_clean < structure.sql
```

### Step 5: Import Data
```bash
# Import all data except inventory
mysql -u root -p handsdb_clean < data_without_inventory.sql

# Import consolidated inventory
mysql -u root -p handsdb_clean -e "
LOAD DATA INFILE '/tmp/inventory_consolidated.csv'
INTO TABLE inventory
FIELDS TERMINATED BY ',' 
ENCLOSED BY '\"'
LINES TERMINATED BY '\n'
(product_id, location_id, quantity, min_quantity, last_updated, @remarks)
SET remarks = NULLIF(@remarks, '');
"
```

### Step 6: Verify
```bash
# Check for duplicates (should return 0 rows)
mysql -u root -p handsdb_clean -e "
SELECT product_id, location_id, COUNT(*) as count
FROM inventory
GROUP BY product_id, location_id
HAVING COUNT(*) > 1;
"

# Show inventory summary
mysql -u root -p handsdb_clean -e "
SELECT 
    p.part_number,
    p.name as product_name,
    l.name as location_name,
    i.quantity,
    i.min_quantity
FROM inventory i
JOIN products p ON i.product_id = p.id
JOIN locations l ON i.location_id = l.id
ORDER BY p.part_number, l.name;
"
```

## Switch to Clean Database

### Option 1: Update Configuration
Edit `v3/config/config.php` and change:
```php
define('DB_NAME', 'handsdb_clean');
```

### Option 2: Rename Databases
```bash
# Backup old database
mysql -u root -p -e "CREATE DATABASE handsdb_old LIKE handsdb"
mysqldump -u root -p handsdb | mysql -u root -p handsdb_old

# Drop old database
mysql -u root -p -e "DROP DATABASE handsdb"

# Rename clean database
mysql -u root -p -e "CREATE DATABASE handsdb LIKE handsdb_clean"
mysqldump -u root -p handsdb_clean | mysql -u root -p handsdb

# Verify
mysql -u root -p handsdb -e "SHOW TABLES"
```

## Troubleshooting

### Error: "Duplicate entry for key 'unique_product_location'"
This means duplicates still exist. Run the consolidation query again:
```sql
CREATE TEMPORARY TABLE temp_inv AS
SELECT 
    product_id,
    location_id,
    SUM(quantity) as quantity,
    MAX(min_quantity) as min_quantity,
    MAX(last_updated) as last_updated,
    GROUP_CONCAT(DISTINCT remarks SEPARATOR '; ') as remarks
FROM inventory
GROUP BY product_id, location_id;

DELETE FROM inventory;

INSERT INTO inventory (product_id, location_id, quantity, min_quantity, last_updated, remarks)
SELECT * FROM temp_inv;

DROP TEMPORARY TABLE temp_inv;
```

### Error: "Can't create database (errno: 13)"
You don't have permission. Try:
```bash
sudo mysql -u root -p
```

### Error: "Access denied for user"
Update the credentials in the scripts or use:
```bash
mysql -u your_username -p
```

## Files Created

- `export_database_with_fix.php` - PHP script to export with consolidation
- `create_clean_database.bat` - Windows batch script for automation
- `complete_export.sql` - Generated SQL file with all data (consolidated)
- `backup_handsdb_YYYYMMDD.sql` - Backup of original database

## What Gets Fixed

1. **Duplicate Inventory Records**: Multiple records for same product-location are consolidated by summing quantities
2. **Unique Constraint**: Ensures `unique_product_location` constraint is properly in place
3. **Data Integrity**: All foreign key relationships are maintained
4. **Remarks**: Multiple remarks are combined with semicolon separator

## Verification Checklist

- [ ] Backup created successfully
- [ ] New database created
- [ ] All tables imported
- [ ] No duplicate inventory records
- [ ] Unique constraint exists on inventory
- [ ] Stock quantities match expected totals
- [ ] Application connects to new database
- [ ] Stock operations work correctly
