<?php

namespace App\Models;

use App\Core\Model;

class EmployeeModel extends Model
{
    protected $table = 'employees';

    /**
     * Get all active employees
     */
    public function getAllEmployees($includeInactive = false)
    {
        $sql = "SELECT * FROM {$this->table}";
        
        if (!$includeInactive) {
            $sql .= " WHERE is_active = 1";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    /**
     * Get employee by ID
     */
    public function getEmployeeById($id)
    {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }

    /**
     * Get employee by employee number
     */
    public function getEmployeeByNumber($employeeNumber)
    {
        $sql = "SELECT * FROM {$this->table} WHERE employee_number = :employee_number";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['employee_number' => $employeeNumber]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }

    /**
     * Create a new employee
     */
    public function createEmployee($data)
    {
        $sql = "INSERT INTO {$this->table} (employee_number, name, department, is_active) 
                VALUES (:employee_number, :name, :department, :is_active)";
        
        $stmt = $this->db->prepare($sql);
        $result = $stmt->execute([
            'employee_number' => $data['employee_number'],
            'name' => $data['name'],
            'department' => $data['department'] ?? null,
            'is_active' => $data['is_active'] ?? 1
        ]);
        
        return $result ? $this->db->lastInsertId() : false;
    }

    /**
     * Update an employee
     */
    public function updateEmployee($id, $data)
    {
        $sql = "UPDATE {$this->table} 
                SET employee_number = :employee_number, 
                    name = :name, 
                    department = :department, 
                    is_active = :is_active 
                WHERE id = :id";
        
        $stmt = $this->db->prepare($sql);
        return $stmt->execute([
            'id' => $id,
            'employee_number' => $data['employee_number'],
            'name' => $data['name'],
            'department' => $data['department'] ?? null,
            'is_active' => $data['is_active'] ?? 1
        ]);
    }

    /**
     * Delete an employee (soft delete by setting is_active = 0)
     */
    public function deleteEmployee($id)
    {
        $sql = "UPDATE {$this->table} SET is_active = 0 WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }

    /**
     * Hard delete an employee
     */
    public function hardDeleteEmployee($id)
    {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }

    /**
     * Bulk create employees
     */
    public function bulkCreateEmployees($employees)
    {
        $created = 0;
        $updated = 0;
        $errors = [];

        foreach ($employees as $index => $employee) {
            try {
                // Check if employee already exists
                $existing = $this->getEmployeeByNumber($employee['employee_number']);
                
                if ($existing) {
                    // Update existing employee
                    $this->updateEmployee($existing['id'], $employee);
                    $updated++;
                } else {
                    // Create new employee
                    $this->createEmployee($employee);
                    $created++;
                }
            } catch (\Exception $e) {
                $errors[] = "Row " . ($index + 1) . ": " . $e->getMessage();
            }
        }

        return [
            'created' => $created,
            'updated' => $updated,
            'errors' => $errors
        ];
    }

    /**
     * Search employees
     */
    public function searchEmployees($query)
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE is_active = 1 
                AND (name LIKE :query OR employee_number LIKE :query)
                ORDER BY name ASC
                LIMIT 20";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['query' => '%' . $query . '%']);
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    /**
     * Get employees by department
     */
    public function getEmployeesByDepartment($department)
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE department = :department AND is_active = 1 
                ORDER BY name ASC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['department' => $department]);
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    /**
     * Get total employee count
     */
    public function getEmployeeCount($includeInactive = false)
    {
        $sql = "SELECT COUNT(*) as count FROM {$this->table}";
        
        if (!$includeInactive) {
            $sql .= " WHERE is_active = 1";
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(\PDO::FETCH_ASSOC);
        return $result['count'];
    }

    /**
     * Check if employee number exists
     */
    public function employeeNumberExists($employeeNumber, $excludeId = null)
    {
        $sql = "SELECT COUNT(*) as count FROM {$this->table} WHERE employee_number = :employee_number";
        $params = ['employee_number' => $employeeNumber];
        
        if ($excludeId) {
            $sql .= " AND id != :exclude_id";
            $params['exclude_id'] = $excludeId;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch(\PDO::FETCH_ASSOC);
        return $result['count'] > 0;
    }
}
