<?php

namespace App\Controllers;

use App\Core\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Cell\DataValidation;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;

class TemplateController extends Controller
{
    public function downloadProductTemplate()
    {
        $this->requireAuth();
        
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Products');
        
        // Set headers with styling - NEW ORDER: Part Number, Product Name, Category, Product Type, Description, Low Stock Threshold, Available Sizes
        $headers = ['Part Number', 'Product Name', 'Category', 'Product Type', 'Description', 'Low Stock Threshold', 'Available Sizes'];
        $sheet->fromArray($headers, null, 'A1');
        
        // Style header row
        $headerStyle = [
            'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '4F46E5']],
            'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
            'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]]
        ];
        $sheet->getStyle('A1:G1')->applyFromArray($headerStyle);
        
        // Set column widths
        $sheet->getColumnDimension('A')->setWidth(20);
        $sheet->getColumnDimension('B')->setWidth(30);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(25);
        $sheet->getColumnDimension('E')->setWidth(40);
        $sheet->getColumnDimension('F')->setWidth(20);
        $sheet->getColumnDimension('G')->setWidth(30);
        
        // Get master data
        $types = $this->getTypes();
        $categories = $this->getCategories();
        $sizes = $this->getSizes();
        
        // Create hidden sheets for dropdowns - use 2D array format
        $categoriesSheet = $spreadsheet->createSheet();
        $categoriesSheet->setTitle('_Categories');
        $categoryData = [];
        foreach ($categories as $cat) {
            $categoryData[] = [$cat['name']];
        }
        if (!empty($categoryData)) {
            $categoriesSheet->fromArray($categoryData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        // Create a sheet for types with category mapping
        $typesSheet = $spreadsheet->createSheet();
        $typesSheet->setTitle('_Types');
        $typeData = [];
        foreach ($types as $type) {
            $typeData[] = [$type['name'], $type['category_name']];
        }
        if (!empty($typeData)) {
            $typesSheet->fromArray($typeData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        $sizesSheet = $spreadsheet->createSheet();
        $sizesSheet->setTitle('_Sizes');
        $sizeData = [];
        foreach ($sizes as $size) {
            $sizeData[] = [$size['size']];
        }
        if (!empty($sizeData)) {
            $sizesSheet->fromArray($sizeData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        // Hide the data sheets
        $categoriesSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        $typesSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        $sizesSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        
        // Add sample data using actual master data values
        $sheet->setCellValue('A2', 'SAMPLE001');
        $sheet->setCellValue('B2', 'Sample Product Name');
        $sheet->setCellValue('C2', count($categories) > 0 ? $categories[0]['name'] : '');
        $sheet->setCellValue('D2', count($types) > 0 ? $types[0]['name'] : '');
        $sheet->setCellValue('E2', 'Sample product description');
        $sheet->setCellValue('F2', '10');
        // For sizes, show example with multiple sizes if available
        if (count($sizes) >= 3) {
            $sheet->setCellValue('G2', $sizes[0]['size'] . ', ' . $sizes[1]['size'] . ', ' . $sizes[2]['size']);
        } elseif (count($sizes) > 0) {
            $sheet->setCellValue('G2', $sizes[0]['size']);
        }
        
        // Calculate counts for validation
        $categoryCount = max(1, count($categories));
        $typeCount = max(1, count($types));
        
        // Add data validation for dropdowns (rows 2-1000)
        for ($row = 2; $row <= 1000; $row++) {
            // Category dropdown
            if (count($categories) > 0) {
                $validation = $sheet->getCell("C{$row}")->getDataValidation();
                $validation->setType(DataValidation::TYPE_LIST);
                $validation->setErrorStyle(DataValidation::STYLE_STOP);
                $validation->setAllowBlank(false);
                $validation->setShowInputMessage(true);
                $validation->setShowErrorMessage(true);
                $validation->setShowDropDown(true);
                $validation->setErrorTitle('Invalid Category');
                $validation->setError('Please select a category from the dropdown');
                $validation->setPromptTitle('Select Category');
                $validation->setPrompt('Choose a category first');
                $validation->setFormula1('_Categories!$A$1:$A$' . $categoryCount);
            }
            
            // Product Type dropdown
            if (count($types) > 0) {
                $validation = $sheet->getCell("D{$row}")->getDataValidation();
                $validation->setType(DataValidation::TYPE_LIST);
                $validation->setErrorStyle(DataValidation::STYLE_STOP);
                $validation->setAllowBlank(false);
                $validation->setShowInputMessage(true);
                $validation->setShowErrorMessage(true);
                $validation->setShowDropDown(true);
                $validation->setErrorTitle('Invalid Type');
                $validation->setError('Please select a type from the dropdown');
                $validation->setPromptTitle('Select Type');
                $validation->setPrompt('Choose a product type that matches your selected category');
                $validation->setFormula1('_Types!$A$1:$A$' . $typeCount);
            }
            
            // Sizes - NO dropdown validation, allow free text for multiple sizes
            // Users can type comma-separated values like "S, M, L, XL"
        }
        
        // Add a visible reference sheet for sizes (so users know what sizes are available)
        $sizeRefSheet = $spreadsheet->createSheet();
        $sizeRefSheet->setTitle('Size Reference');
        $sizeRefSheet->setCellValue('A1', 'Available Sizes');
        $sizeRefSheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
        $sizeRefSheet->setCellValue('A2', 'Use these size names in the "Available Sizes" column.');
        $sizeRefSheet->setCellValue('A3', 'For multiple sizes, separate with commas (e.g., S, M, L, XL)');
        $sizeRefSheet->getStyle('A2:A3')->getFont()->setItalic(true);
        
        // List all available sizes
        $sizeRefSheet->setCellValue('A5', 'Size Name');
        $sizeRefSheet->getStyle('A5')->getFont()->setBold(true);
        $sizeRefSheet->getStyle('A5')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E5E7EB');
        
        $rowNum = 6;
        foreach ($sizes as $size) {
            $sizeRefSheet->setCellValue('A' . $rowNum, $size['size']);
            $rowNum++;
        }
        $sizeRefSheet->getColumnDimension('A')->setWidth(25);
        
        // Add instructions sheet
        $instructionsSheet = $spreadsheet->createSheet();
        $instructionsSheet->setTitle('Instructions');
        $instructionsSheet->setCellValue('A1', 'Product Bulk Upload Instructions');
        $instructionsSheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
        $instructionsSheet->setCellValue('A3', '1. Fill in the "Products" sheet with your product data');
        $instructionsSheet->setCellValue('A4', '2. Part Number: Unique identifier for each product (e.g., SKU)');
        $instructionsSheet->setCellValue('A5', '3. Product Name: Descriptive name for the product');
        $instructionsSheet->setCellValue('A6', '4. Category: Select from dropdown (required)');
        $instructionsSheet->setCellValue('A7', '5. Product Type: Select from dropdown - must match the selected category');
        $instructionsSheet->setCellValue('A8', '6. Description: Brief description of the product');
        $instructionsSheet->setCellValue('A9', '7. Low Stock Threshold: Number to trigger alerts (default: 5)');
        $instructionsSheet->setCellValue('A10', '8. Available Sizes: Type size names separated by commas (e.g., S, M, L, XL)');
        $instructionsSheet->setCellValue('A11', '   - See the "Size Reference" sheet for available size names');
        $instructionsSheet->setCellValue('A12', '   - For "One Size" products, just type: One Size');
        $instructionsSheet->setCellValue('A14', 'IMPORTANT: Make sure the Product Type you select belongs to the Category you selected.');
        $instructionsSheet->setCellValue('A15', 'Note: Delete the sample row before uploading');
        $instructionsSheet->getColumnDimension('A')->setWidth(90);
        
        $spreadsheet->setActiveSheetIndex(0);
        
        // Output file
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="products_template.xlsx"');
        header('Cache-Control: max-age=0');
        
        $writer = new Xlsx($spreadsheet);
        $writer->save('php://output');
        exit;
    }
    
    public function downloadStockTemplate()
    {
        $this->requireAuth();
        
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Stock');
        
        // Set headers with styling - NEW: Product Name instead of Part Number
        $headers = ['Product Name', 'Location', 'Size', 'Quantity', 'Remarks', 'Seller', 'Price Per Unit'];
        $sheet->fromArray($headers, null, 'A1');
        
        // Style header row
        $headerStyle = [
            'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '10B981']],
            'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
            'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]]
        ];
        $sheet->getStyle('A1:G1')->applyFromArray($headerStyle);
        
        // Set column widths
        $sheet->getColumnDimension('A')->setWidth(30);
        $sheet->getColumnDimension('B')->setWidth(25);
        $sheet->getColumnDimension('C')->setWidth(15);
        $sheet->getColumnDimension('D')->setWidth(15);
        $sheet->getColumnDimension('E')->setWidth(30);
        $sheet->getColumnDimension('F')->setWidth(20);
        $sheet->getColumnDimension('G')->setWidth(15);
        
        // Get master data
        $products = $this->model('ProductModel')->getAll();
        $locations = $this->getLocations();
        $sizes = $this->getSizes();
        $sellers = $this->getSellers();
        
        // Create hidden sheets for dropdowns - use 2D array format for proper column population
        // Store both product name and part_number for lookup
        $productsSheet = $spreadsheet->createSheet();
        $productsSheet->setTitle('_Products');
        $productData = [];
        foreach ($products as $p) {
            // Column A: Product Name (for dropdown), Column B: Part Number (for reference)
            $productData[] = [$p['name'], $p['part_number']];
        }
        if (!empty($productData)) {
            $productsSheet->fromArray($productData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        $locationsSheet = $spreadsheet->createSheet();
        $locationsSheet->setTitle('_Locations');
        $locationData = [];
        foreach ($locations as $loc) {
            $locationData[] = [$loc['name']];
        }
        if (!empty($locationData)) {
            $locationsSheet->fromArray($locationData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        $sizesSheet = $spreadsheet->createSheet();
        $sizesSheet->setTitle('_Sizes');
        $sizeData = [];
        foreach ($sizes as $size) {
            $sizeData[] = [$size['size']];
        }
        if (!empty($sizeData)) {
            $sizesSheet->fromArray($sizeData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        $sellersSheet = $spreadsheet->createSheet();
        $sellersSheet->setTitle('_Sellers');
        $sellerData = [];
        foreach ($sellers as $seller) {
            $sellerData[] = [$seller['name']];
        }
        if (!empty($sellerData)) {
            $sellersSheet->fromArray($sellerData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        // Hide the data sheets
        $productsSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        $locationsSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        $sizesSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        $sellersSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        
        // Add sample data using actual master data values
        $sheet->setCellValue('A2', count($products) > 0 ? $products[0]['name'] : 'Sample Product Name');
        $sheet->setCellValue('B2', count($locations) > 0 ? $locations[0]['name'] : 'Warehouse A');
        $sheet->setCellValue('C2', count($sizes) > 0 ? $sizes[0]['size'] : 'M');
        $sheet->setCellValue('D2', '100');
        $sheet->setCellValue('E2', 'Initial stock');
        $sheet->setCellValue('F2', count($sellers) > 0 ? $sellers[0]['name'] : '');
        $sheet->setCellValue('G2', '25.50');
        
        // Add data validation for dropdowns (rows 2-1000)
        $productCount = max(1, count($products));
        $locationCount = max(1, count($locations));
        $sizeCount = max(1, count($sizes));
        $sellerCount = max(1, count($sellers));
        
        for ($row = 2; $row <= 1000; $row++) {
            // Product Name dropdown
            if (count($products) > 0) {
                $validation = $sheet->getCell("A{$row}")->getDataValidation();
                $validation->setType(DataValidation::TYPE_LIST);
                $validation->setErrorStyle(DataValidation::STYLE_STOP);
                $validation->setAllowBlank(false);
                $validation->setShowInputMessage(true);
                $validation->setShowErrorMessage(true);
                $validation->setShowDropDown(true);
                $validation->setErrorTitle('Invalid Product');
                $validation->setError('Please select a product from the dropdown');
                $validation->setPromptTitle('Select Product');
                $validation->setPrompt('Choose a product by name');
                $validation->setFormula1('_Products!$A$1:$A$' . $productCount);
            }
            
            // Location dropdown
            if (count($locations) > 0) {
                $validation = $sheet->getCell("B{$row}")->getDataValidation();
                $validation->setType(DataValidation::TYPE_LIST);
                $validation->setErrorStyle(DataValidation::STYLE_STOP);
                $validation->setAllowBlank(false);
                $validation->setShowInputMessage(true);
                $validation->setShowErrorMessage(true);
                $validation->setShowDropDown(true);
                $validation->setErrorTitle('Invalid Location');
                $validation->setError('Please select a location from the dropdown');
                $validation->setPromptTitle('Select Location');
                $validation->setPrompt('Choose a warehouse location');
                $validation->setFormula1('_Locations!$A$1:$A$' . $locationCount);
            }
            
            // Size dropdown
            if (count($sizes) > 0) {
                $validation = $sheet->getCell("C{$row}")->getDataValidation();
                $validation->setType(DataValidation::TYPE_LIST);
                $validation->setErrorStyle(DataValidation::STYLE_WARNING);
                $validation->setAllowBlank(true);
                $validation->setShowInputMessage(true);
                $validation->setShowErrorMessage(true);
                $validation->setShowDropDown(true);
                $validation->setErrorTitle('Invalid Size');
                $validation->setError('Please select a size from the dropdown');
                $validation->setPromptTitle('Select Size');
                $validation->setPrompt('Choose a size (leave blank if not applicable)');
                $validation->setFormula1('_Sizes!$A$1:$A$' . $sizeCount);
            }
            
            // Seller dropdown
            if (count($sellers) > 0) {
                $validation = $sheet->getCell("F{$row}")->getDataValidation();
                $validation->setType(DataValidation::TYPE_LIST);
                $validation->setErrorStyle(DataValidation::STYLE_WARNING);
                $validation->setAllowBlank(true);
                $validation->setShowInputMessage(true);
                $validation->setShowErrorMessage(true);
                $validation->setShowDropDown(true);
                $validation->setErrorTitle('Invalid Seller');
                $validation->setError('Please select a seller from the dropdown');
                $validation->setPromptTitle('Select Seller');
                $validation->setPrompt('Choose a seller (optional)');
                $validation->setFormula1('_Sellers!$A$1:$A$' . $sellerCount);
            }
        }
        
        // Add a reference sheet for products (showing product name and part number)
        $productRefSheet = $spreadsheet->createSheet();
        $productRefSheet->setTitle('Product Reference');
        $productRefSheet->setCellValue('A1', 'Product Name');
        $productRefSheet->setCellValue('B1', 'Part Number');
        $productRefSheet->getStyle('A1:B1')->getFont()->setBold(true)->setSize(12);
        $productRefSheet->getStyle('A1:B1')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E5E7EB');
        
        $rowNum = 2;
        foreach ($products as $product) {
            $productRefSheet->setCellValue('A' . $rowNum, $product['name']);
            $productRefSheet->setCellValue('B' . $rowNum, $product['part_number']);
            $rowNum++;
        }
        $productRefSheet->getColumnDimension('A')->setWidth(35);
        $productRefSheet->getColumnDimension('B')->setWidth(20);
        
        // Add instructions sheet
        $instructionsSheet = $spreadsheet->createSheet();
        $instructionsSheet->setTitle('Instructions');
        $instructionsSheet->setCellValue('A1', 'Stock Bulk Upload Instructions');
        $instructionsSheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
        $instructionsSheet->setCellValue('A3', '1. Fill in the "Stock" sheet with your stock data');
        $instructionsSheet->setCellValue('A4', '2. Product Name: Select from dropdown (required) - see "Product Reference" sheet for all products');
        $instructionsSheet->setCellValue('A5', '3. Location: Select warehouse location from dropdown (required)');
        $instructionsSheet->setCellValue('A6', '4. Size: Select from dropdown (leave blank if not applicable)');
        $instructionsSheet->setCellValue('A7', '5. Quantity: Number of items to add (required)');
        $instructionsSheet->setCellValue('A8', '6. Remarks: Optional notes');
        $instructionsSheet->setCellValue('A9', '7. Seller: Select from dropdown (optional)');
        $instructionsSheet->setCellValue('A10', '8. Price Per Unit: Optional price information');
        $instructionsSheet->setCellValue('A12', 'Note: Delete the sample row before uploading');
        $instructionsSheet->setCellValue('A13', 'Tip: Check the "Product Reference" sheet to see all available products with their part numbers');
        $instructionsSheet->getColumnDimension('A')->setWidth(90);
        
        $spreadsheet->setActiveSheetIndex(0);
        
        // Output file
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="stock_template.xlsx"');
        header('Cache-Control: max-age=0');
        
        $writer = new Xlsx($spreadsheet);
        $writer->save('php://output');
        exit;
    }
    
    private function getTypes()
    {
        $sql = "SELECT t.id, t.name, c.name as category_name 
                FROM types t
                LEFT JOIN categories c ON t.category_id = c.id
                ORDER BY c.name, t.name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    private function getCategories()
    {
        $sql = "SELECT id, name FROM categories ORDER BY name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    private function getSizes()
    {
        $sql = "SELECT id, size FROM product_sizes ORDER BY id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    private function getLocations()
    {
        $sql = "SELECT id, name FROM locations ORDER BY name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    private function getSellers()
    {
        $sql = "SELECT id, name FROM sellers ORDER BY name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    public function downloadEmployeeTemplate()
    {
        $this->requireAuth();
        
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Employees');
        
        // Set headers with styling
        $headers = ['Employee Number', 'Name', 'Department', 'Is Active'];
        $sheet->fromArray($headers, null, 'A1');
        
        // Style header row
        $headerStyle = [
            'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '6366F1']],
            'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
            'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]]
        ];
        $sheet->getStyle('A1:D1')->applyFromArray($headerStyle);
        
        // Set column widths
        $sheet->getColumnDimension('A')->setWidth(20);
        $sheet->getColumnDimension('B')->setWidth(35);
        $sheet->getColumnDimension('C')->setWidth(25);
        $sheet->getColumnDimension('D')->setWidth(15);
        
        // Get master data
        $departments = $this->getDepartments();
        
        // Create hidden sheet for departments
        $departmentsSheet = $spreadsheet->createSheet();
        $departmentsSheet->setTitle('_Departments');
        $departmentData = [];
        foreach ($departments as $dept) {
            $departmentData[] = [$dept['name']];
        }
        if (!empty($departmentData)) {
            $departmentsSheet->fromArray($departmentData, null, 'A1');
        }
        $spreadsheet->setActiveSheetIndex(0);
        
        // Hide the data sheet
        $departmentsSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        
        // Add sample data using actual master data values
        $sheet->setCellValue('A2', 'EMP001');
        $sheet->setCellValue('B2', 'John Doe');
        $sheet->setCellValue('C2', count($departments) > 0 ? $departments[0]['name'] : 'Sample Department');
        $sheet->setCellValue('D2', 'Yes');
        
        // Add data validation for dropdowns (rows 2-1000)
        $departmentCount = max(1, count($departments));
        
        for ($row = 2; $row <= 1000; $row++) {
            // Department dropdown
            if (count($departments) > 0) {
                $validation = $sheet->getCell("C{$row}")->getDataValidation();
                $validation->setType(DataValidation::TYPE_LIST);
                $validation->setErrorStyle(DataValidation::STYLE_WARNING);
                $validation->setAllowBlank(true);
                $validation->setShowInputMessage(true);
                $validation->setShowErrorMessage(true);
                $validation->setShowDropDown(true);
                $validation->setErrorTitle('Invalid Department');
                $validation->setError('Please select a department from the dropdown');
                $validation->setPromptTitle('Select Department');
                $validation->setPrompt('Choose a department (optional)');
                $validation->setFormula1('_Departments!$A$1:$A$' . $departmentCount);
            }
            
            // Is Active dropdown
            $validation = $sheet->getCell("D{$row}")->getDataValidation();
            $validation->setType(DataValidation::TYPE_LIST);
            $validation->setErrorStyle(DataValidation::STYLE_STOP);
            $validation->setAllowBlank(false);
            $validation->setShowInputMessage(true);
            $validation->setShowErrorMessage(true);
            $validation->setShowDropDown(true);
            $validation->setErrorTitle('Invalid Status');
            $validation->setError('Please select Yes or No');
            $validation->setPromptTitle('Select Status');
            $validation->setPrompt('Is this employee active?');
            $validation->setFormula1('"Yes,No"');
        }
        
        // Add instructions sheet
        $instructionsSheet = $spreadsheet->createSheet();
        $instructionsSheet->setTitle('Instructions');
        $instructionsSheet->setCellValue('A1', 'Employee Bulk Upload Instructions');
        $instructionsSheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
        $instructionsSheet->setCellValue('A3', '1. Fill in the "Employees" sheet with your employee data');
        $instructionsSheet->setCellValue('A4', '2. Employee Number: Unique identifier for each employee (required)');
        $instructionsSheet->setCellValue('A5', '3. Name: Full name of the employee (required)');
        $instructionsSheet->setCellValue('A6', '4. Department: Select from dropdown (optional)');
        $instructionsSheet->setCellValue('A7', '5. Is Active: Select Yes or No (required, default: Yes)');
        $instructionsSheet->setCellValue('A9', 'Note: Delete the sample row before uploading');
        $instructionsSheet->setCellValue('A10', 'Tip: If an employee number already exists, the record will be updated');
        $instructionsSheet->getColumnDimension('A')->setWidth(90);
        
        $spreadsheet->setActiveSheetIndex(0);
        
        // Output file
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="employees_template.xlsx"');
        header('Cache-Control: max-age=0');
        
        $writer = new Xlsx($spreadsheet);
        $writer->save('php://output');
        exit;
    }
    
    private function getDepartments()
    {
        $sql = "SELECT id, name FROM departments ORDER BY name";
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll();
    }
}
